/**
 * Copyright (c) 2021 Tencent. All rights reserved.
 * Module:   TRTCCloud @ TXLiteAVSDK
 * Function: 腾讯云 TRTC 主功能接口
 * Version: <:Version:>
 */
using System;
using UnityEngine;

namespace trtc {

    public abstract class ITRTCCloud : IDeprecatedTRTCCloud {
        public static string PLUGIN_VERSION = "13.0.0.20258_389";

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    创建实例和事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 1.1 创建 TRTCCloud 实例（单例模式）。
         *
         * @param context 仅适用于 Android 平台，SDK 内部会将其转化为 Android 平台的 ApplicationContext 用于调用 Android System API。
         *        如果传入的 `context` 参数为空，SDK 内部会自动获取当前进程的 ApplicationContext。
         * @note
         * 1. 如果您使用 `delete ITRTCCloud*` 会导致编译错误，请使用 {@link destroyTRTCShareInstance} 释放对象指针。
         * 2. 在 Windows、Mac 和 iOS 平台上，请调用 `getTRTCShareInstance()` 接口。
         * 3. 在 Android 平台上，请调用 `getTRTCShareInstance(void *context)` 接口。
         */
        public static ITRTCCloud getTRTCShareInstance() {
            return TRTCCloudImplement.getShareInstance();
        }

        /**
         * 1.2 销毁 TRTCCloud 实例（单例模式）。
         */
        public static void destroyTRTCShareInstance() {
            TRTCCloudImplement.destroyShareInstance();
        }

        /**
         * 1.3 添加 TRTC 事件回调。
         *
         * 您可以通过 {@link TRTCCloudCallback} 获得来自 SDK 的各类事件通知（比如：错误码，警告码，音视频状态参数等）。
         */
        public abstract void addCallback(ITRTCCloudCallback callback);

        /**
         * 1.4 移除 TRTC 事件回调。
         */
        public abstract void removeCallback(ITRTCCloudCallback callback);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    房间相关接口函数
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 2.1 进入房间。
         *
         * TRTC 的所有用户都需要进入房间才能“发布”或“订阅”音视频流，“发布”是指将自己的音频和视频推送到云端，“订阅”是指从云端拉取房间里其他用户的音视频流。
         * 调用该接口时，您需要指定您的应用场景 {@link TRTCAppScene} 以获取最佳的音视频传输体验，这些场景可以分成两大类：
         * **实时通话：**
         * 包括 {@link TRTCAppSceneVideoCall} 和 {@link TRTCAppSceneAudioCall} 两个可选项，分别是视频通话和语音通话，该模式适合 1对1 的音视频通话，或者参会人数在 300 人以内的在线会议。
         * **在线直播：**
         * 包括 {@link TRTCAppSceneLIVE} 和 {@link TRTCAppSceneVoiceChatRoom} 两个可选项，分别是视频直播和语音直播，该模式适合十万人以内的在线直播场景，但需要您在接下来介绍的 {@link TRTCParams} 参数中指定 **角色(role)**
         * 这个字段，也就是将房间中的用户区分为 **主播** ({@link TRTCRoleAnchor}) 和 **观众** ({@link TRTCRoleAudience}) 两种不同的角色。 调用该接口后，您会收到来自 {@link TRTCCloudCallback} 中的 {@link onEnterRoom} 回调：
         *  - 如果进房成功，参数 `result` 会是一个正数（result > 0），表示从函数调用到进入房间所花费的时间，单位是毫秒（ms）。
         *  - 如果进房失败，参数 `result` 会是一个负数（result < 0），表示进房失败的[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param param 进房参数，用于指定用户的身份、角色和安全票据等信息，详情请参见  {@link TRTCParams} 。
         * @param scene 应用场景，用于指定您的业务场景，同一个房间内的所有用户需要设定相同的 {@link TRTCAppScene}。
         * @note
         *   1. 同一个房间内的所有用户需要设定相同的 `scene`。不同的 `scene` 会导致偶现的异常问题。
         *   2. 当您指定参数 `scene` 为 {@link TRTCAppSceneLIVE} 或 {@link TRTCAppSceneVoiceChatRoom} 时，您必须通过 {@link TRTCParams} 中的 `role`字段为当前用户设定他/她在房间中的角色。
         *   3. 请您尽量保证 {@link enterRoom} 与 {@link exitRoom} 前后配对使用，即保证“先退出前一个房间再进入下一个房间”，否则会导致很多异常问题。
         */
        public abstract void enterRoom(ref TRTCParams param, TRTCAppScene scene);

        /**
         * 2.2 离开房间。
         *
         * 调用该接口会让用户离开自己所在的音视频房间，并释放摄像头、麦克风、扬声器等设备资源。
         * 等资源释放完毕之后，SDK 会通过 {@link TRTCCloudCallback} 中的 {@link onExitRoom} 回调向您通知。
         * 如果您要再次调用  {@link enterRoom}  或者切换到其他的供应商的 SDK，建议等待 {@link onExitRoom} 回调到来之后再执行之后的操作，以避免摄像头或麦克风被占用的问题。
         */
        public abstract void exitRoom();
        public abstract void exitRoom(bool stopLocalVideo, bool stopLocalAudio);

        /**
         * 2.3 切换角色。
         *
         * 调用本接口可以实现用户在“主播”和“观众”两种角色之间来回切换。
         * 由于视频直播和语音聊天室需要支持多达10万名观众同时观看，所以设定了 **只有主播才能发布自己的音视频** 的规则。因此，当有些观众希望发布自己的音视频流（以便能跟主播互动）时，就需要先把自己的角色切换成 **主播**。
         * 您可以在进入房间时通过 {@link TRTCParams} 中的 `role` 字段事先确定用户的角色，也可以在进入房间后通过该接口动态切换角色。
         * @param role 角色，默认为 **主播**。
         * - {@link TRTCRoleAnchor} ：主播，可以发布自己的音视频，同一个房间里最多支持50个主播同时发布音视频。
         * - {@link TRTCRoleAudience} ：观众，不能发布自己的音视频流，只能观看房间中其他主播的音视频。如果要发布自己的音视频，需要先通过 {@link switchRole} 切换成 **主播**，同一个房间内同时最多可以容纳 10 万名观众。
         * @note
         * 1. 该接口仅适用于视频直播（{@link TRTCAppSceneLIVE}）和语音聊天室（{@link TRTCAppSceneVoiceChatRoom}）这两个场景。
         * 2. 如果您在调用 {@link enterRoom} 时指定的 `scene` 为 {@link TRTCAppSceneVideoCall} 或 {@link TRTCAppSceneAudioCall}，请不要调用这个接口。
         */
        public abstract void switchRole(TRTCRoleType role);

        /**
         * 2.5 切换房间。
         *
         * 使用该接口可以让用户快速从一个房间切换到另一个房间。
         * - 如果用户的身份是“观众”，该接口的调用效果等同于 `exitRoom(当前房间) + enterRoom（新的房间）`。
         * - 如果用户的身份是“主播”，该接口在切换房间的同时还会保持自己的音视频发布状态，因此在房间切换过程中，摄像头的预览和声音的采集都不会中断。
         * 该接口适用于在线教育场景中，监课老师在多个房间中进行快速切换的场景。在该场景下使用 `switchRoom` 可以获得比 `exitRoom + enterRoom` 更好的流畅性和更少的代码量。
         * 接口调用结果会通过 {@link TRTCCloudCallback} 中的 {@link onSwitchRoom} 回调。
         * @param config 房间参数，详情请参见 {@link TRTCSwitchRoomConfig}。
         * @note 由于对老版本 SDK 兼容的需求，参数 `config` 中同时包含 `roomId` 与 `strRoomId` 两个参数，这两个参数的填写格外讲究，请注意如下事项：
         *  1. 若您选用 `strRoomId`，则 `roomId` 需要填写为0。若两者都填，将优先选用 `roomId`。
         *  2. 所有房间需要同时使用 `strRoomId` 或同时使用 `roomId`，不可混用，否则将会出现很多预期之外的 bug。
         */
        public abstract void switchRoom(TRTCSwitchRoomConfig config);

        /**
         * 2.6 请求跨房通话。
         *
         * 默认情况下，只有同一个房间中的用户之间可以进行音视频通话，不同的房间之间的音视频流是相互隔离的。
         * 但您可以通过调用该接口，将另一个房间中某个主播音视频流发布到自己所在的房间中，与此同时，该接口也会将自己的音视频流发布到目标主播的房间中。
         * 也就是说，您可以使用该接口让身处两个不同房间中的主播进行跨房间的音视频流分享，从而让每个房间中的观众都能观看到这两个主播的音视频。该功能可以用来实现主播之间的 PK 功能。
         * 跨房通话的请求结果会通过 {@link TRTCCloudCallback} 中的 {@link onConnectOtherRoom} 回调通知给您。
         * 例如：当房间“101”中的主播 A 通过 `connectOtherRoom()` 跟房间“102”中的主播 B 建立跨房通话后，
         * - 房间“101”中的用户都会收到主播 B 的 `onRemoteUserEnterRoom(B)` 和 `onUserVideoAvailable(B,true)` 这两个事件回调，即房间“101”中的用户都可以订阅主播 B 的音视频。
         * - 房间“102”中的用户都会收到主播 A 的 `onRemoteUserEnterRoom(A)` 和 `onUserVideoAvailable(A,true)` 这两个事件回调，即房间“102”中的用户都可以订阅主播 A 的音视频。
         * ![](https://qcloudimg.tencent-cloud.cn/raw/c5e6c72fc163ad5c0b6b7b00e1da55b5.png)
         * 跨房通话的参数考虑到后续扩展字段的兼容性问题，暂时采用了 JSON 格式的参数：
         * **情况一：数字房间号**
         * 如果房间“101”中的主播 A 要跟房间“102”中的主播 B 连麦，那么主播 A 调用该接口时需要传入：`{"roomId": 102, "userId": "userB"}`
         * 示例代码如下：
         * **情况二：字符串房间号**
         * 如果您使用的是字符串房间号，务必请将 json 中的 “roomId” 替换成 “strRoomId”: `{"strRoomId": "102", "userId": "userB"}`
         * 示例代码如下：
         * @param param 需要你传入 JSON 格式的字符串参数，`roomId` 代表数字格式的房间号，`strRoomId` 代表字符串格式的房间号，`userId` 代表目标主播的用户 ID。
         */
        public abstract void connectOtherRoom(string jsonParams);

        /**
         * 2.7 退出跨房通话。
         *
         * 退出结果会通过 {@link TRTCCloudCallback} 中的 {@link onDisconnectOtherRoom} 回调通知给您。
         */
        public abstract void disconnectOtherRoom();

        /**
         * 2.8 设置订阅模式（需要在进入房前设置才能生效）。
         *
         * 您可以通过该接口在“自动订阅”和“手动订阅”两种模式下进行切换：
         * - 自动订阅：默认模式，用户在进入房间后会立刻接收到该房间中的音视频流，音频会自动播放，视频会自动开始解码（依然需要您通过 {@link startRemoteView} 接口绑定渲染控件）。
         * - 手动订阅：在用户进入房间后，需要手动调用 {@link startRemoteView} 接口才能启动视频流的订阅和解码，需要手动调用 {@link muteRemoteAudio} (false) 接口才能启动声音的播放。
         * 在绝大多数场景下，用户进入房间后都会订阅房间中所有主播的音视频流，因此 TRTC 默认采用了自动订阅模式，以求得最佳的“秒开体验”。
         * 如果您的应用场景中每个房间同时会有很多路音视频流在发布，而每个用户只想选择性地订阅其中的 1-2 路，则推荐使用“手动订阅”模式以节省流量费用。
         * @param autoRecvAudio true：自动订阅音频；false：需手动调用 `muteRemoteAudio(false)` 订阅音频。默认值：true。
         * @param autoRecvVideo true：自动订阅视频；false：需手动调用 `startRemoteView` 订阅视频。默认值：true。
         * @note
         * 1. 需要在进入房间前调用该接口进行设置才能生效。
         * 2. 在自动订阅模式下，如果用户在进入房间后没有调用  {@link startRemoteView} 订阅视频流，SDK 会自动停止订阅视频流，以便达到节省流量的目的。
         */
        public abstract void setDefaultStreamRecvMode(bool autoRecvAudio, bool autoRecvVideo);

        /**
         * 2.9 创建子房间实例（用于多房间并发观看）。
         *
         * TRTCCloud 一开始被设计成单例模式，限制了多房间并发观看的能力。
         * 通过调用该接口，您可以创建出多个 TRTCCloud 实例，以便同时进入多个不同的房间观看音视频流。
         * 但需要注意的是，您在多个 TRTCCloud 实例中发布音视频流的能力会受到一定限制。
         * 该功能主要用于在线教育场景中一种被称为“超级小班课”的业务场景中，用于解决“每个 TRTC 的房间中最多只能有 50 人同时发布自己音视频流”的限制。
         * 示例代码如下：
         * @note
         * 1. 同一个用户，可以使用同一个 userId 进入多个不同 roomId 的房间。
         * 2. 两台不同的终端设备不可以同时使用同一个 userId 进入同一个 roomId 的房间。
         * 3. 您可以分别为不同实例分别设置 {@link TRTCCloudCallback} 获取各自的事件通知。
         * 4. 同一个用户可以在多个 TRTCCloud 实例中推流，也可以调用子实例中与本地音视频相关的接口。但需要注意：
         *    - 多实例的音频需要同时为麦克风采集或自定义采集，而且与音频设备相关的接口调用会以最后一次为准；
         *    - 与摄像头相关的调用会以最后一次为准：{@link startLocalPreview}。
         * @return 子 TRTCCloud 实例
         */
        public abstract ITRTCCloud createSubCloud();

        /**
         * 2.10 销毁子房间实例。
         *
         * @param subCloud 子房间实例
         */
        public abstract void destroySubCloud(ITRTCCloud subCloud);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    CDN 相关接口函数
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 3.1  开始发布媒体流。
         *
         * 该接口会向 TRTC 服务器发送指令，要求其将当前用户的音视频流转推/转码到直播 CDN 或者回推到 TRTC 房间中您可以通过 {@link TRTCPublishTarget} 配置中的 {@link TRTCPublishMode} 指定具体的发布模式
         * @param target 媒体流发布的目标地址，具体配置参考 {@link TRTCPublishTarget}。支持转推/转码到腾讯或者第三方 CDN，也支持转码回推到 TRTC 房间中。
         * @param params 媒体流编码输出参数，具体配置参考 {@link TRTCStreamEncoderParam}。转码和回推到 TRTC 房间中时为必填项，您需要指定您预期的转码输出参数。在转推时，为了更好的转推稳定性和 CDN 兼容性，也建议您进行配置。
         * @param config 媒体流转码配置参数。具体配置参考 {@link TRTCStreamMixingConfig}。转码和回推到 TRTC 房间中时为必填项，您需要指定您预期的转码配置参数。转推模式下则无效。
         * @note
         * 1. SDK 会通过回调 {@link onStartPublishMediaStream} 带给您后台启动的任务标识（即 taskId）。
         * 2. 同一个任务（{@link TRTCPublishMode} 与 {@link TRTCPublishCdnUrl} 均相同）仅支持启动一次。若您后续需要更新或者停止该项任务，需要记录并使用返回的 taskId，通过 {@link updatePublishMediaStream} 或者 {@link stopPublishMediaStream} 来操作。
         * 3. `target` 支持同时配置多个 CDN URL（最多同时 10 个）。若您的同一个转推/转码任务需要发布至多路 CDN，则仅需要在 `target` 中配置多个 CDN URL 即可。同一个转码任务即使有多个转推地址，对应的转码计费仍只收取一份。
         * 4. 使用时需要注意不要多个任务同时往相同的 URL 地址推送，以免引起异常推流状态。一种推荐的方案是 URL 中使用 “sdkappid_roomid_userid_main” 作为区分标识，这种命名方式容易辨认且不会在您的多个应用中发生冲突。
         */
        public abstract void startPublishMediaStream(ref TRTCPublishTarget target, ref TRTCStreamEncoderParam param, ref TRTCStreamMixingConfig config);

        /**
         * 3.2 更新发布媒体流。
         *
         * 该接口会向 TRTC 服务器发送指令，更新通过 {@link startPublishMediaStream} 启动的媒体流
         * @param taskId 通过回调 {@link onStartPublishMediaStream} 带给您后台启动的任务标识（即 taskId）
         * @param target 媒体流发布的目标地址，具体配置参考 {@link TRTCPublishTarget}。支持转推/转码到腾讯或者第三方 CDN，也支持转码回推到 TRTC 房间中。
         * @param params 媒体流编码输出参数，具体配置参考 {@link TRTCStreamEncoderParam}。转码和回推到 TRTC 房间中时为必填项，您需要指定您预期的转码输出参数。在转推时，为了更好的转推稳定性和 CDN 兼容性，也建议您进行配置。
         * @param config 媒体流转码配置参数。具体配置参考 {@link TRTCStreamMixingConfig}。转码和回推到 TRTC 房间中时为必填项，您需要指定您预期的转码配置参数。转推模式下则无效。
         * @note
         * 1. 您可以通过本接口来更新发布的 CDN URL（支持增删，最多同时 10 个），但您使用时需要注意不要多个任务同时往相同的 URL 地址推送，以免引起异常推流状态。
         * 2. 您可以通过 `taskId` 来更新调整转推/转码任务。例如在 pk 业务中，您可以先通过 {@link startPublishMediaStream} 发起转推，接着在主播发起 pk 时，通过 `taskId` 和本接口将转推更新为转码任务。此时，CDN
         * 播放将连续并且不会发生断流（您需要保持媒体流编码输出参数 param 一致）。
         * 3. 同一个任务不支持纯音频、音视频、纯视频之间的切换。
         */
        public abstract void updatePublishMediaStream(string taskId, ref TRTCPublishTarget target, ref TRTCStreamEncoderParam param, ref TRTCStreamMixingConfig config);

        /**
         * 3.3 停止发布媒体流。
         *
         * 该接口会向 TRTC 服务器发送指令，停止通过 {@link startPublishMediaStream} 启动的媒体流
         * @param taskId 通过回调 {@link onStartPublishMediaStream} 带给您后台启动的任务标识（即 taskId）
         * @note
         * 1. 若您的业务后台并没有保存该 `taskId`，在您的主播异常退房重进后，如果您需要重新获取 `taskId`，您可以再次调用 {@link startPublishMediaStream} 启动任务。此时 TRTC 后台会返回任务启动失败，同时带给您上一次启动的 `taskId`。
         * 2. 若 `taskId` 填空字符串，将会停止该用户所有通过 {@link startPublishMediaStream} 启动的媒体流，如果您只启动了一个媒体流或者想停止所有通过您启动的媒体流，推荐使用这种方式。
         */
        public abstract void stopPublishMediaStream(string taskId);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    视频相关接口函数
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 4.1 开启本地摄像头的预览画面（移动端）。
         *
         * 在 {@link enterRoom} 之前调用此函数，SDK 只会开启摄像头，并一直等到您调用 {@link enterRoom} 之后才开始推流。
         * 在 {@link enterRoom} 之后调用此函数，SDK 会开启摄像头并自动开始视频推流。
         * 当开始渲染首帧摄像头画面时，您会收到 {@link TRTCCloudCallback} 中的 {@link onCameraDidReady} 回调通知。
         * @param frontCamera true：前置摄像头；false：后置摄像头。
         * @param view 承载视频画面的控件。
         * @note 如果希望开播前预览摄像头画面并通过 BeautyManager 调节美颜参数，您可以：
         * - 方案一：在调用 {@link enterRoom} 之前调用 `startLocalPreview`。
         * - 方案二：在调用 {@link enterRoom} 之后调用 `startLocalPreview + muteLocalVideo(true)`。
         */
        public abstract void startLocalPreview(bool frontCamera, GameObject view);

        /**
         * 4.4 停止摄像头预览。
         */
        public abstract void stopLocalPreview();

        /**
         * 4.5 暂停/恢复发布本地的视频流。
         *
         * 该接口可以暂停（或恢复）发布本地的视频画面，暂停之后，同一房间中的其他用户将无法继续看到自己画面。
         * 该接口在指定 {@link TRTCVideoStreamTypeBig} 时等效于 `startLocalPreview + stopLocalPreview` 这两个接口，但具有更好的响应速度。
         * 因为 `startLocalPreview + stopLocalPreview` 需要打开和关闭摄像头，而打开和关闭摄像头都是硬件设备相关的操作，非常耗时。
         * 相比之下，`muteLocalVideo` 只需要在软件层面对数据流进行暂停或者放行即可，因此效率更高，也更适合需要频繁打开关闭的场景。
         * 当暂停/恢复发布指定 {@link TRTCVideoStreamTypeBig} 后，同一房间中的其他用户将会收到 {@link onUserVideoAvailable} 回调通知。
         * 当暂停/恢复发布指定 {@link TRTCVideoStreamTypeSub} 后，同一房间中的其他用户将会收到 {@link onUserSubStreamAvailable} 回调通知。
         * @param streamType 要暂停/恢复的视频流类型（仅支持 {@link TRTCVideoStreamTypeBig} 和 {@link TRTCVideoStreamTypeSub}）。
         * @param mute true：暂停；false：恢复。
         */
        public abstract void muteLocalVideo(TRTCVideoStreamType streamType, bool mute);

        /**
         * 4.7 订阅远端用户的视频流，并绑定视频渲染控件。
         *
         * 调用该接口可以让 SDK 拉取指定 userId 的视频流，并渲染到参数 `view` 指定的渲染控件上。您可以通过 {@link setRemoteRenderParams} 设置画面的显示模式。
         * - 如果您已经知道房间中有视频流的用户的 userId，可以直接调用 `startRemoteView` 订阅该用户的画面。
         * - 如果您不知道房间中有哪些用户在发布视频，您可以在 {@link enterRoom} 之后等待来自 {@link onUserVideoAvailable} 的通知。
         * 调用本接口只是启动视频流的拉取，此时画面还需要加载和缓冲，当缓冲完毕后您会收到来自 {@link onFirstVideoFrame} 的通知。
         * @param userId 指定远端用户的 ID。
         * @param streamType 指定要观看 userId 的视频流类型。
         *    - 高清大画面：{@link TRTCVideoStreamTypeBig}。
         *    - 低清小画面：{@link TRTCVideoStreamTypeSmall}（需要远端用户通过 {@link enableSmallVideoStream} 开启双路编码后才有效果）。
         *    - 辅流画面（常用于屏幕分享）：{@link TRTCVideoStreamTypeSub}。
         * @param view 用于承载视频画面的渲染控件。
         * @note 注意几点规则需要您关注：
         *  1. SDK 支持同时观看某 userId 的大画面和辅路画面，或者同时观看某 userId 的小画面和辅路画面，但不支持同时观看大画面和小画面。
         *  2. 只有当指定的 userId 通过 {@link enableSmallVideoStream} 开启双路编码后，才能观看该用户的小画面。
         *  3. 当指定的 userId 的小画面不存在时，SDK 默认切换到该用户的大画面。
         */
        public abstract void startRemoteView(string userId, TRTCVideoStreamType streamType, GameObject view);

        /**
         * 4.9 停止订阅远端用户的视频流，并释放渲染控件。
         *
         * 调用此接口会让 SDK 停止接收该用户的视频流，并释放该路视频流的解码和渲染资源。
         * @param userId 指定远端用户的 ID。
         * @param streamType 指定要观看 userId 的视频流类型。
         * - 高清大画面：{@link TRTCVideoStreamTypeBig}。
         * - 低清小画面：{@link TRTCVideoStreamTypeSmall}。
         * - 辅流画面（常用于屏幕分享）：{@link TRTCVideoStreamTypeSub}。
         */
        public abstract void stopRemoteView(string userId, TRTCVideoStreamType streamType);

        /**
         * 4.10 停止订阅所有远端用户的视频流，并释放全部渲染资源。
         *
         * 调用此接口会让 SDK 停止接收所有来自远端的视频流，并释放全部的解码和渲染资源。
         * @note 如果当前有正在显示的辅路画面（屏幕分享）也会一并被停止。
         */
        public abstract void stopAllRemoteView();

        /**
         * 4.11 暂停/恢复订阅远端用户的视频流。
         *
         * 该接口仅暂停/恢复接收指定用户的视频流，但并不释放显示资源，视频画面会被冻屏在接口调用时的最后一帧。
         * @param userId 指定远端用户的 ID。
         * @param streamType 要暂停/恢复的视频流类型。
         * - 高清大画面：{@link TRTCVideoStreamTypeBig}。
         * - 低清小画面：{@link TRTCVideoStreamTypeSmall}。
         * - 辅流画面（常用于屏幕分享）：{@link TRTCVideoStreamTypeSub}。
         * @param mute 是否暂停接收。
         * @note 该接口支持您在进入房间（{@link enterRoom}）前调用，暂停状态在退出房间（{@link exitRoom}）之后会被重置。调用该接口暂停接收指定用户的视频流后，只调用 {@link startRemoteView} 接口无法播放指定用户的视频，需要调用 {@link
         * muteRemoteVideoStream}(false) 或 {@link muteAllRemoteVideoStreams}(false) 才能恢复。
         */
        public abstract void muteRemoteVideoStream(string userId, TRTCVideoStreamType streamType, bool mute);

        /**
         * 4.12 暂停/恢复订阅所有远端用户的视频流。
         *
         * 该接口仅暂停/恢复接收所有用户的视频流，但并不释放显示资源，视频画面会被冻屏在接口调用时的最后一帧。
         * @param mute  是否暂停接收。
         * @note 该接口支持您在进入房间（{@link enterRoom}）前调用，暂停状态在退出房间（{@link exitRoom}）之后会被重置。
         * @note 调用该接口暂停接收所有用户的视频流后，只调用 {@link startRemoteView} 接口无法播放指定用户的视频，需要调用 {@link muteRemoteVideoStream}(false) 或 {@link muteAllRemoteVideoStreams}(false) 才能恢复。
         */
        public abstract void muteAllRemoteVideoStreams(bool mute);

        /**
         * 4.13 设置视频编码器的编码参数。
         *
         * 该设置能够决定远端用户看到的画面质量，同时也能决定云端录制出的视频文件的画面质量。
         * @param param 用于设置视频编码器的相关参数，详情请参见 {@link TRTCVideoEncParam}。
         * @note 从v11.5版本开始，编码输出分辨率会按照宽8高2字节对齐，并且是向下调整，eg:输入分辨率540x960，实际编码输出分辨率536x960。
         */
        public abstract void setVideoEncoderParam(ref TRTCVideoEncParam param);

        /**
         * 4.14 设置网络质量控制的相关参数。
         *
         * 该设置决定在差网络环境下的质量调控策略，如“画质优先”或“流畅优先”等策略。
         * @param param 用于设置网络质量控制的相关参数，详情请参见 {@link TRTCNetworkQosParam}。
         */
        public abstract void setNetworkQosParam(ref TRTCNetworkQosParam param);

        /**
         * 4.15 设置本地画面的渲染参数。
         *
         * 可设置的参数包括有：画面的旋转角度、填充模式以及左右镜像等。
         * @param params 画面渲染参数，详情请参见 {@link TRTCRenderParams}。
         */
        public abstract void setLocalRenderParams(TRTCRenderParams renderParams);

        /**
         * 4.16 设置远端画面的渲染模式。
         *
         * 可设置的参数包括有：画面的旋转角度、填充模式以及左右镜像等。
         * @param userId 指定远端用户的 ID。
         * @param streamType 可以设置为主路画面（{@link TRTCVideoStreamTypeBig}）或辅路画面（{@link TRTCVideoStreamTypeSub}）。
         * @param params 画面渲染参数，详情请参见 {@link TRTCRenderParams}。
         */
        public abstract void setRemoteRenderParams(string userId, TRTCVideoStreamType streamType, ref TRTCRenderParams renderParams);

        /**
         * 4.20 开启大小画面双路编码模式。
         *
         * 开启双路编码模式后，当前用户的编码器会同时输出【高清大画面】和【低清小画面】两路视频流（但只有一路音频流）。
         * 如此以来，房间中的其他用户就可以根据自身的网络情况或屏幕大小选择订阅【高清大画面】或是【低清小画面】。
         * @param enable 是否开启小画面编码，默认值：false。
         * @param smallVideoEncParam 小流的视频参数。
         * @return 0：成功；-1：当前大画面已被设置为较低画质，开启双路编码已无必要。
         * @note 双路编码开启后，会消耗更多的 CPU 和 网络带宽，所以 Mac、Windows 或者高性能 Pad 可以考虑开启，不建议手机端开启。
         */
        public abstract void enableSmallVideoStream(bool enable, ref TRTCVideoEncParam smallVideoParam);

        /**
         * 4.21 切换指定远端用户的大小画面。
         *
         * 当房间中某个主播开启了双路编码之后，房间中其他用户通过 {@link startRemoteView} 订阅到的画面默认会是【高清大画面】。
         * 您可以通过此接口选定希望订阅的画面是大画面还是小画面，该接口在 {@link startRemoteView} 之前和之后调用均可生效。
         * @param userId 指定远端用户的 ID。
         * @param streamType 视频流类型，即选择看大画面还是小画面，默认为大画面。
         * @note 此功能需要目标用户已经通过 {@link enableSmallVideoStream} 提前开启了双路编码模式，否则此调用无实际效果。
         */
        public abstract void setRemoteVideoStreamType(string userId, TRTCVideoStreamType type);

        /**
         * 4.25 设置重力感应的适配模式（11.7 及以上版本）。
         *
         * 开启重力感应后，如果采集端的设备发生旋转，采集端和观众端的画面都会进行相应地渲染以确保视野中的画面始终朝上。
         * 只在sdk内部摄像头采集场景生效，并且只在移动端生效。
         * 1. 该接口仅对采集端起作用，如果只是观看房间中的画面，开启此接口是无效的
         * 2. 当采集端设备发生 90 度或 270 度旋转时，采集端或者观众看到的画面可能会被裁剪以保持比例的协调
         * @param mode 重力感应模式，详情请参见 {@link TRTCGravitySensorAdaptiveMode_Disable}、{@link TRTCGravitySensorAdaptiveMode_FillByCenterCrop} 和 {@link TRTCGravitySensorAdaptiveMode_FitWithBlackBorder} 默认值：{@link
         * TRTCGravitySensorAdaptiveMode_Disable}。
         */
        public abstract void setGravitySensorAdaptiveMode(TRTCGravitySensorAdaptiveMode mode);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    音频相关接口函数
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 5.1 开启本地音频的采集和发布。
         *
         * SDK 默认不开启麦克风，当用户需要发布本地音频时，需要调用该接口开启麦克风采集，并将音频编码并发布到当前的房间中。
         * 开启本地音频的采集和发布后，房间中的其他用户会收到 {@link onUserAudioAvailable}(userId, true) 的通知。
         * @param quality 声音音质
         * - {@link TRTCAudioQualitySpeech}，流畅：单声道；音频裸码率：18kbps；适合语音通话为主的场景，比如在线会议，语音通话。
         * - {@link TRTCAudioQualityDefault}，默认：单声道；音频裸码率：50kbps；SDK 默认的音频质量，如无特殊需求推荐选择之。
         * - {@link TRTCAudioQualityMusic}，高音质：双声道 + 全频带；音频裸码率：128kbps；适合需要高保真传输音乐的场景，比如在线K歌、音乐直播等。
         * @note 该函数会检查麦克风的使用权限，如果当前 App 没有麦克风权限，SDK 会自动向用户申请麦克风使用权限。
         */
        public abstract void startLocalAudio(TRTCAudioQuality quality);

        /**
         * 5.2 停止本地音频的采集和发布。
         *
         * 停止本地音频的采集和发布后，房间中的其他用户会收到 {@link onUserAudioAvailable}(userId, false) 的通知。
         */
        public abstract void stopLocalAudio();

        /**
         * 5.3 暂停/恢复发布本地的音频流。
         *
         * 当您暂停发布本地音频流之后，房间中的其他用户会收到 {@link onUserAudioAvailable}(userId, false) 的通知。
         * 当您恢复发布本地音频流之后，房间中的其他用户会收到 {@link onUserAudioAvailable}(userId, true) 的通知。
         * 与 {@link stopLocalAudio} 的不同之处在于，`muteLocalAudio(true)` 并不会释放麦克风权限，而是继续发送码率极低的静音包。
         * 这对于需要云端录制的场景非常适用，因为 MP4 等格式的视频文件，对于音频数据的连续性要求很高，使用 {@link stopLocalAudio} 会导致录制出的 MP4 文件不易播放。
         * 因此在对录制文件的质量要求较高的场景中，建议选择 `muteLocalAudio` 而不建议使用 {@link stopLocalAudio}。
         * @param mute true：静音；false：恢复。
         */
        public abstract void muteLocalAudio(bool mute);

        /**
         * 5.4 暂停/恢复播放远端的音频流。
         *
         * 当您静音某用户的远端音频时，SDK 会停止播放指定用户的声音，同时也会停止拉取该用户的音频数据。
         * @param userId 用于指定远端用户的 ID。
         * @param mute true：静音；false：取消静音。
         * @note 在进入房间（{@link enterRoom}）之前或之后调用本接口均生效，静音状态在退出房间（{@link exitRoom}）之后会被重置为 false。
         */
        public abstract void muteRemoteAudio(string userId, bool mute);

        /**
         * 5.5 暂停/恢复播放所有远端用户的音频流。
         *
         * 当您静音所有用户的远端音频时，SDK 会停止播放所有来自远端的音频流，同时也会停止拉取所有用户的音频数据。
         * @param mute true：静音；false：取消静音。
         * @note 在进入房间（{@link enterRoom}）之前或之后调用本接口均生效，静音状态在退出房间（{@link exitRoom}）之后会被重置为 false。
         */
        public abstract void muteAllRemoteAudio(bool mute);

        /**
         * 5.7 设定某一个远端用户的声音播放音量。
         *
         * 您可以通过 `setRemoteAudioVolume(userId, 0)` 将某一个远端用户的声音静音。
         * @param userId 用于指定远端用户的 ID。
         * @param volume 音量大小，取值范围为 [0, 150]，默认值：100。
         * @note 如果将 `volume` 设置成 100 之后感觉音量还是太小，可以将 `volume` 最大设置成 150，但超过 100 的 `volume` 会有爆音的风险，请谨慎操作。
         */
        public abstract void setRemoteAudioVolume(string userId, int volume);

        /**
         * 5.8 设定本地音频的采集音量。
         *
         * @param volume 音量大小，取值范围为 [0, 150]；默认值：100。
         * @note 如果将 `volume` 设置成 100 之后感觉音量还是太小，可以将 `volume` 最大设置成 150，但超过 100 的 `volume` 会有爆音的风险，请谨慎操作。
         */
        public abstract void setAudioCaptureVolume(int volume);

        /**
         * 5.9 获取本地音频的采集音量。
         *
         * @return 采集音量。
         */
        public abstract int getAudioCaptureVolume();

        /**
         * 5.10 设定远端音频的播放音量。
         *
         * 该接口会控制 SDK 最终交给系统播放的声音音量，调节效果会影响到本地音频录制文件的音量大小，但不会影响到耳返的音量大小。
         * @param volume 音量大小，取值范围为 [0, 150]，默认值：100。
         * @note 如果将 volume 设置成 100 之后感觉音量还是太小，可以将 volume 最大设置成 150，但超过 100 的 volume 会有爆音的风险，请谨慎操作。
         */
        public abstract void setAudioPlayoutVolume(int volume);

        /**
         * 5.11 获取远端音频的播放音量。
         */
        public abstract int getAudioPlayoutVolume();

        /**
         * 5.12 启用音量大小提示。
         *
         * 开启此功能后，SDK 会在 {@link TRTCCloudCallback} 中的 {@link onUserVoiceVolume} 回调中反馈本地和远端用户的音频音量评估信息，包括音量大小、人声检测、音频频谱等。
         * @note 如需启用音量大小提示，请在调用 {@link startLocalAudio} 之前调用该接口.
         * @param enable 是否启用音量提示，默认为关闭状态。
         * @param params 音量评估等相关参数，请参见 {@link TRTCAudioVolumeEvaluateParams}。
         */
        public abstract void enableAudioVolumeEvaluation(bool enable, TRTCAudioVolumeEvaluateParams evaluateParams);

        /**
         * 5.15 开启本地媒体录制。
         *
         * 开启后把直播过程中的音视频内容录制到本地的一个文件中，该功能不会产生额外费用。
         * @param params 录制参数，请参见 {@link TRTCLocalRecordingParams}。
         */
        public abstract void startLocalRecording(ref TRTCLocalRecordingParams localRecordingParams);

        /**
         * 5.16 停止本地媒体录制。
         *
         * 如果录制任务在退出房间前尚未通过本接口停止，则退出房间后录音任务会自动被停止。
         */
        public abstract void stopLocalRecording();

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    设备管理相关接口
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 6.1 获取设备管理类（TXDeviceManager）。
         *
         * 通过设备管理，您可以设置摄像头、麦克风和扬声器等音视频相关的硬件设备功能。
         * @return 设备管理类 {@link TXDeviceManager}。
         */
        public abstract ITXDeviceManager getDeviceManager();

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    美颜特效和图像水印
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 7.1 设置美颜、美白、红润等特效。
         *
         * SDK 内部集成了两套风格不同的磨皮算法：
         * -“光滑”：算法比较激进，磨皮效果比较明显，适用于秀场直播。
         * -“自然”：算法更多地保留了面部细节，磨皮效果更加自然，适用于绝大多数直播场景。
         * @param style 磨皮算法，有“光滑”和“自然”两种算法。
         * @param beautyLevel 美颜级别，取值范围[0, 9]，0表示关闭，1 - 9值越大，效果越明显。
         * @param whitenessLevel 美白级别，取值范围[0, 9]，0表示关闭，1 - 9值越大，效果越明显。
         * @param ruddinessLevel 红润级别，取值范围[0, 9]，0表示关闭，1 - 9值越大，效果越明显。
         */
        public abstract void setBeautyStyle(TRTCBeautyStyle style, uint beauty, uint white, uint ruddiness);

        /**
         * 7.2 添加水印。
         *
         * 水印的位置是通过 xOffset, yOffset, fWidthRatio 来指定的。
         * - xOffset：水印的坐标，取值范围为 [0, 1] 的浮点数。
         * - yOffset：水印的坐标，取值范围为 [0, 1] 的浮点数。
         * - fWidthRatio：水印的大小比例，取值范围为 [0, 1] 的浮点数。
         * @param streamType 要设置水印的流类型（{@link TRTCVideoStreamTypeBig}，{@link TRTCVideoStreamTypeSub}）。
         * @param srcData    水印图片源数据（传 nullptr 表示去掉水印）。
         * @param srcType    水印图片源数据类型。请参见 {@link TRTCWaterMarkSrcType}。
         * @param nWidth     水印图片像素宽度（源数据为文件路径时忽略该参数）。
         * @param nHeight    水印图片像素高度（源数据为文件路径时忽略该参数）。
         * @param xOffset    水印显示的左上角 x 轴偏移。
         * @param yOffset    水印显示的左上角 y 轴偏移。
         * @param fWidthRatio 水印显示的宽度占画面宽度比例（水印按该参数等比例缩放显示）。
         * @param isVisibleOnLocalPreview true：本地预览显示水印；false：本地预览不显示水印，仅在win/mac下生效。
         * @note 本接口只支持给主路视频添加图片水印。
         */
        public abstract void setWaterMark(TRTCVideoStreamType streamType, string srcData, TRTCWaterMarkSrcType srcType, uint nWidth, uint nHeight, float xOffset, float yOffset, float fWidthRatio, bool isVisibleOnLocalPreview = false);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    背景音乐和声音特效
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 8.1 获取音效管理类（TXAudioEffectManager）。
         *
         * TXAudioEffectManager 是音效管理接口，您可以通过该接口实现如下功能：
         * - 背景音乐：支持在线音乐和本地音乐，支持变速、变调等特效、支持原生和伴奏并播放和循环播放。
         * - 耳机耳返：麦克风捕捉的声音实时通过耳机播放，常用于音乐直播。
         * - 混响效果：KTV、小房间、大会堂、低沉、洪亮等。
         * - 变声特效：萝莉、大叔、重金属等。
         * - 短音效：鼓掌声、欢笑声等简短的音效文件（对于小于10秒的文件，请将 isShortFile 参数设置为 true）。
         * @return 音效管理类 {@link TXAudioEffectManager}。
         */
        public abstract ITXAudioEffectManager getAudioEffectManager();

        /**
         * 8.2 开启系统声音采集（iOS 端暂未支持）。
         *
         * 该接口会从电脑的声卡中采集音频数据，并将其混入到 SDK 当前的音频数据流中，从而使房间中的其他用户也能听到主播的电脑所播放出的声音。
         * 在线教育场景中，老师可以使用此功能让 SDK 采集教学影片中的声音，并广播给同房间中的学生。
         * 音乐直播场景中，主播可以使用此功能让 SDK 采集音乐播放器中的音乐，从而为自己的直播间增加背景音乐。
         * @param deviceName 您可以指定该参数为空值（nullptr），代表让 SDK 采集整个系统的声音。
         * @note
         * 在 Windows 平台下，您也可以将参数 `deviceName` 设置为某个应用程序的可执行文件（如 QQMuisc.exe）的绝对路径，此时 SDK 只会采集该应用程序的声音（支持 32 位版本的 SDK，64 位版本的 SDK 需要满足 Windows 版本为 10.0.19042 或更高版本）。
         * 您也可以指定该参数为某个扬声器设备的名称来采集特定扬声器声音（通过接口 {@link TXDeviceManager} 中的 getDevicesList 接口，可以获取类型为 {@link TXMediaDeviceTypeSpeaker} 的扬声器设备）。
         * 在 Windows 平台下您也可以指定该参数为某个进程的进程 id（格式为 “process_xxx”，其中 xxx 为进程 id），此时 SDK 知会采集该进程的声音（需要满足 Windows 版本为 10.0.19042或更高版本）。
         * 在 Windows 平台下您也可以指定该参数为某个进程的进程 id（格式为 “exclude_process_xxx”，其中 xxx 为进程 id），此时 SDK 会采集除了该进程的其他所有声音（需要满足 Windows 版本为 10.0.19042或更高版本）。
         */
        public abstract void startSystemAudioLoopback(string deviceName);

        /**
         * 8.3 停止系统声音采集（iOS 端暂未支持）。
         */
        public abstract void stopSystemAudioLoopback();

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    屏幕分享相关接口
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 9.1 启动屏幕分享。
         *
         * 该接口可以抓取整个屏幕的内容，或抓取您指定的某个应用的窗口内容，并将其分享给同房间中的其他用户。
         * @param view 渲染控件所在的父控件，可以设置为空值，表示不显示屏幕分享的预览效果。
         * @param streamType 屏幕分享使用的线路，可以设置为主路（TRTCVideoStreamTypeBig）或者辅路（TRTCVideoStreamTypeSub），推荐使用辅路。
         * @param encParam 屏幕分享的画面编码参数，SDK 会优先使用您通过此接口设置的编码参数：
         *   - 如果您设置 `encParam` 为空值，且您已通过 {@link setSubStreamEncoderParam} 设置过辅路视频编码参数，SDK 将使用您设置过的辅路编码参数进行屏幕分享。
         *   - 如果您设置 `encParam` 为空值，且您未通过 {@link setSubStreamEncoderParam} 设置过辅路视频编码参数，SDK 将自动选择一个最佳的编码参数进行屏幕分享。
         * @note
         * 1. 同一个用户同时最多只能发布一路主路（{@link TRTCVideoStreamTypeBig}）画面和一路辅路（{@link TRTCVideoStreamTypeSub}）画面。
         * 2. 默认情况下，屏幕分享使用辅路画面。如果使用主路做屏幕分享，您需要提前停止摄像头采集（{@link stopLocalPreview}）以避免相互冲突。
         * 3. 同一个房间中同时只能有一个用户使用辅路做屏幕分享，也就是说，同一个房间中同时只允许一个用户开启辅路。
         * 4. 当房间中已经有其他用户在使用辅路分享屏幕时，此时调用该接口会收到来自 {@link TRTCCloudCallback} 的 `onError(ERR_SERVER_CENTER_ANOTHER_USER_PUSH_SUB_VIDEO)` 回调。
         */
        public abstract void startScreenCapture(GameObject view, TRTCVideoStreamType type, ref TRTCVideoEncParam param);

        /**
         * 9.2 停止屏幕分享。
         */
        public abstract void stopScreenCapture();

        /**
         * 9.3 暂停屏幕分享。
         *
         * @note 从v11.5版本开始，暂停屏幕采集会使用最后一帧按照 1 fps 帧率输出。
         */
        public abstract void pauseScreenCapture();

        /**
         * 9.4 恢复屏幕分享。
         */
        public abstract void resumeScreenCapture();

        /**
         * 9.5 枚举可分享的屏幕和窗口（该接口仅支持桌面系统）。
         *
         * 当您在对接桌面端系统的屏幕分享功能时，一般都需要展示一个选择分享目标的界面，这样用户能够使用这个界面选择是分享整个屏幕还是某个窗口。
         * 通过本接口，您就可以查询到当前系统中可用于分享的窗口的 ID、名称以及缩略图。我们在 Demo 中提供了一份默认的界面实现供您参考。
         * @param thumbnailSize 指定要获取的窗口缩略图大小，缩略图可用于绘制在窗口选择界面上。
         * @param iconSize 指定要获取的窗口图标大小。
         * @return 窗口列表包括屏幕。
         * @note
         * 1. 返回的列表中包含屏幕和应用窗口，屏幕是列表中的第一个元素。如果用户有多个显示器，那么每个显示器都是一个分享目标。
         * 2. 请不要使用 `delete ITRTCScreenCaptureSourceList*` 删除 SourceList，这很容易导致崩溃，请使用 {@link ITRTCScreenCaptureSourceList} 中的 release 方法释放列表。
         */
        public abstract TRTCScreenCaptureSourceInfo[] getScreenCaptureSources(SIZE thumbnailSize, SIZE iconSize);

        /**
         * 9.6 选取要分享的屏幕或窗口（该接口仅支持桌面系统）。
         *
         * 当您通过 {@link getScreenCaptureSources} 获取到可以分享的屏幕和窗口之后，您可以调用该接口选定期望分享的目标屏幕或目标窗口。
         * 在屏幕分享的过程中，您也可以随时调用该接口以切换分享目标。
         * 支持如下四种情况：
         * - 共享整个屏幕：source 中 type 为 {@link TRTCScreenCaptureSourceTypeScreen} 的 source，captureRect 设为 { 0, 0, 0, 0 }。
         * - 共享指定区域：source 中 type 为 {@link TRTCScreenCaptureSourceTypeScreen} 的 source，captureRect 设为非 nullptr，例如 { 100, 100, 300, 300 }。
         * - 共享整个窗口：source 中 type 为 {@link TRTCScreenCaptureSourceTypeWindow} 的 source，captureRect 设为 { 0, 0, 0, 0 }。
         * - 共享窗口区域：source 中 type 为 {@link TRTCScreenCaptureSourceTypeWindow} 的 source，captureRect 设为非 nullptr，例如 { 100, 100, 300, 300 }。
         * @param source        指定分享源。详情参考 {@link TRTCScreenCaptureSourceInfo}
         * @param captureRect   指定捕获的区域。
         * @param property      指定屏幕分享目标的属性，包括捕获鼠标，高亮捕获窗口等，详情参考 {@link TRTCScreenCaptureProperty}。
         * @note 设置高亮边框颜色、宽度参数在 Mac 平台不生效。
         */
        public abstract void selectScreenCaptureTarget(TRTCScreenCaptureSourceInfo source, Rect captureRect, TRTCScreenCaptureProperty property);

        /**
         * 9.7 设置屏幕分享（即辅路）的视频编码参数（桌面系统和移动系统均已支持）。
         *
         * 该接口可以设定远端用户所看到的屏幕分享（即辅路）的画面质量，同时也能决定云端录制出的视频文件中屏幕分享的画面质量。
         * 请注意如下两个接口的差异：
         * - {@link setVideoEncoderParam} 用于设置主路画面（{@link TRTCVideoStreamTypeBig}，一般用于摄像头）的视频编码参数。
         * - {@link setSubStreamEncoderParam} 用于设置辅路画面（{@link TRTCVideoStreamTypeSub}，一般用于屏幕分享）的视频编码参数。
         * @param param 辅流编码参数，详情请参见 {@link TRTCVideoEncParam}。
         */
        public abstract void setSubStreamEncoderParam(ref TRTCVideoEncParam param);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    自定义采集和自定义渲染
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 10.1 启用/关闭视频自定义采集模式。
         *
         * 开启该模式后，SDK 不再运行原有的视频采集流程，即不再继续从摄像头采集数据和美颜，而是只保留视频编码和发送能力。
         * 您需要通过 {@link sendCustomVideoData} 不断地向 SDK 塞入自己采集的视频画面。
         * @param streamType 用于指定视频流类型，{@link TRTCVideoStreamTypeBig}：高清大画面；{@link TRTCVideoStreamTypeSub}：辅路画面。
         * @param enable 是否启用视频自定义采集，默认值：false。
         */
        public abstract void enableCustomVideoCapture(TRTCVideoStreamType streamType, bool enable);

        /**
         * 10.2 向 SDK 投送自己采集的视频帧。
         *
         * 使用此接口可以向 SDK 投送自己采集的视频帧，SDK 会将视频帧进行编码并通过自身的网络模块传输出去。
         * 参数 {@link TRTCVideoFrame} 推荐下列填写方式（其他字段不需要填写）：
         * - pixelFormat：Windows 和 Android 平台仅支持 {@link TRTCVideoPixelFormat_I420}，iOS 和 Mac平台支持 {@link TRTCVideoPixelFormat_I420} 和 {@link TRTCVideoPixelFormat_BGRA32}。
         * - bufferType：推荐选择 {@link TRTCVideoBufferType_PixelBuffer}。
         * - data：用于承载视频帧数据的 buffer。
         * - length：视频帧数据长度，如果 pixelFormat 设定为 I420 格式，length 可以按照如下公式计算：`length = width × height × 3 / 2`。
         * - width：视频图像的宽度，如 640 px。
         * - height：视频图像的高度，如 480 px。
         * - timestamp：时间戳，单位为毫秒（ms），请使用视频帧在采集时被记录下来的时间戳（可以在采集到一帧视频帧之后，通过调用 {@link generateCustomPTS} 获取时间戳）。
         *
         * 参考文档：[自定义采集和渲染](https://cloud.tencent.com/document/product/647/34066)。
         * @param streamType 用于指定视频流类型，{@link TRTCVideoStreamTypeBig}： 高清大画面；{@link TRTCVideoStreamTypeSub}： 辅路画面。
         * @param frame 视频数据，支持 I420 格式数据。
         * @note
         * 1. 推荐您在采集到的一帧视频帧后，即调用 {@link generateCustomPTS} 接口获取该帧的 timestamp 数值，这样可以获得最佳的音画同步效果。
         * 2. SDK 最终编码出的视频帧率并不是由您调用本接口的频率决定的，而是由您在 {@link setVideoEncoderParam} 中所设置的 FPS 决定的。
         * 3. 请尽量保持本接口的调用间隔是均匀的，否则会导致编码器输出帧率不稳或者音画不同步等问题。
         * 4. iOS 和 Mac平台目前支持传入 {@link TRTCVideoPixelFormat_I420} 或 {@link TRTCVideoPixelFormat_BGRA32} 格式的视频帧。
         * 5. Windows 和 Android 平台目前仅支持传入 {@link TRTCVideoPixelFormat_I420} 格式的视频帧。
         */
        public abstract void sendCustomVideoData(TRTCVideoStreamType streamType, TRTCVideoFrame frame);

        /**
         * 10.3 启用音频自定义采集模式。
         *
         * 开启该模式后，SDK 不再运行原有的音频采集流程，即不再继续从麦克风采集音频数据，而是只保留音频编码和发送能力。
         * 您需要通过 {@link sendCustomAudioData} 不断地向 SDK 塞入自己采集的音频数据。
         * @param enable 是否启用音频自定义采集模式，默认值：false。
         * @note 由于回声抵消（AEC）需要严格的控制声音采集和播放的时间，所以开启自定义音频采集后，AEC 能力可能会失效。
         */
        public abstract void enableCustomAudioCapture(bool enable);

        /**
         * 10.4 向 SDK 投送自己采集的音频数据。
         *
         * 参数 {@link TRTCAudioFrame} 推荐下列填写方式（其他字段不需要填写）：
         * - audioFormat：音频数据格式，仅支持 TRTCAudioFrameFormatPCM。
         * - data：音频帧 buffer。音频帧数据只支持 PCM 格式，支持 5-100 ms 帧长，推荐使用 20ms 帧长，长度计算方法：48000采样率，单声道的帧长度：`48000 × 0.02s × 1 × 16bit = 15360bit = 1920字节`。
         * - sampleRate：采样率，支持：16000、24000、32000、44100、48000。
         * - channel：声道数（如果是立体声，数据是交叉的），单声道：1； 双声道：2。
         * - timestamp：时间戳，单位为毫秒（ms），请使用音频帧在采集时被记录下来的时间戳（可以在采集到一帧音频帧之后，通过调用 {@link generateCustomPTS} 获取时间戳）。
         *
         * 参考文档：[自定义采集和渲染](https://cloud.tencent.com/document/product/647/74692)。
         * @param frame 音频数据
         * @note 请您精准地按每帧时长的间隔调用本接口，数据投送间隔不均匀时极易触发声音卡顿。
         */
        public abstract void sendCustomAudioData(TRTCAudioFrame frame);

        /**
         * 10.5 启用/关闭自定义音轨。
         *
         * 开启后，您可以通过本接口向 SDK 混入一条自定义的音轨。通过两个布尔型参数，您可以控制该音轨是否要在远端和本地播放。
         * @param enablePublish 控制混入的音轨是否要在远端播放，默认值：false。
         * @param enablePlayout 控制混入的音轨是否要在本地播放，默认值：false。
         * @note 如果您指定参数 `enablePublish` 和 `enablePlayout` 均为 false，代表完全关闭您的自定义音轨。
         */
        public abstract void enableMixExternalAudioFrame(bool enablePublish, bool enablePlayout);

        /**
         * 10.9.1 开启视频第三方美颜。
         *
         * 开启后，您可以通过 {@link ITRTCVideoFrameCallback} 获取到指定像素格式和视频数据结构类型的图像帧。
         * @param enable      是否启用视频第三方美颜，默认为关闭状态。
         * @param pixelFormat 指定回调的像素格式。
         * @param bufferType  指定视频数据结构类型。
         * @return 0：成功；<0：错误
         */
        public abstract int enableLocalVideoCustomProcess(bool enable, TRTCVideoPixelFormat pixelFormat, TRTCVideoBufferType bufferType);

        /**
         * 10.9.2 设置第三方美颜的视频数据回调。
         *
         * 设置该回调之后，SDK 会把采集到的视频帧通过您设置的 callback 回调出来，用于第三方美颜组件进行二次处理，之后 SDK 会将处理后的视频帧进行编码和发送。
         * @param callback 自定义美颜回调，详见 {@link ITRTCVideoFrameCallback}。
         */
        public abstract void setLocalVideoCustomProcessCallback(ITRTCVideoFrameCallback callback);

        /**
         * 10.10 设置本地视频自定义渲染回调。
         *
         * 设置该回调之后，SDK 内部会跳过原来的渲染流程，并把采集到的数据回调出来，您需要自己完成画面渲染。
         * - 您可以通过调用 `setLocalVideoRenderCallback(TRTCVideoPixelFormat_Unknown, TRTCVideoBufferType_Unknown, nullptr)` 停止回调。
         * - iOS、Mac、Windows 平台目前仅支持回调 {@link TRTCVideoPixelFormat_I420} 或 {@link TRTCVideoPixelFormat_BGRA32} 像素格式的视频帧。
         * - Android 平台目前仅支持传入 {@link TRTCVideoPixelFormat_I420}, {@link TRTCVideoPixelFormat_RGBA32} 或 {@link TRTCVideoPixelFormat_Texture_2D} 像素格式的视频帧。
         * @param pixelFormat 指定回调的像素格式
         * @param bufferType  指定视频数据结构类型
         * @param callback    自定义渲染回调
         * @return 0：成功；<0：错误（详见 [错误码](https://cloud.tencent.com/document/product/647/38308)）。
         */
        public abstract int setLocalVideoRenderCallback(TRTCVideoPixelFormat pixelFormat, TRTCVideoBufferType bufferType, ITRTCVideoRenderCallback callback);

        /**
         * 10.11 设置远端视频自定义渲染回调。
         *
         * 设置该回调之后，SDK 内部会跳过原来的渲染流程，并把采集到的数据回调出来，您需要自己完成画面渲染。
         * - 您可以通过调用 `setRemoteVideoRenderCallback(TRTCVideoPixelFormat_Unknown, TRTCVideoBufferType_Unknown, nullptr)` 停止回调。
         * - iOS、Mac、Windows 平台目前仅支持回调 {@link TRTCVideoPixelFormat_I420} 或 {@link TRTCVideoPixelFormat_BGRA32} 像素格式的视频帧。
         * - Android 平台目前仅支持传入 {@link TRTCVideoPixelFormat_I420}, {@link TRTCVideoPixelFormat_RGBA32} 或 {@link TRTCVideoPixelFormat_Texture_2D} 像素格式的视频帧。
         *
         * @note 实际使用时，需要先调用 startRemoteView(userid, nullptr) 来获取远端用户的视频流（view 设置为 nullptr 即可），否则不会有数据回调出来。
         * @param userId 远端用户id
         * @param pixelFormat 指定回调的像素格式
         * @param bufferType  指定视频数据结构类型，目前只支持 {@link TRTCVideoBufferType_Buffer}
         * @param callback    自定义渲染回调。
         * @return 0：成功；< 0：错误（详见 [错误码](https://cloud.tencent.com/document/product/647/38308)）。
         */
        public abstract int setRemoteVideoRenderCallback(string userId, TRTCVideoPixelFormat pixelFormat, TRTCVideoBufferType bufferType, ITRTCVideoRenderCallback callback);

        /**
         * 10.12 设置音频数据自定义回调。
         *
         * 设置该回调之后，SDK 内部会把音频数据（PCM 格式）回调出来，包括：
         * - {@link onCapturedAudioFrame}：本地麦克风采集到的音频数据回调
         * - {@link onLocalProcessedAudioFrame}：本地采集并经过音频模块前处理后的音频数据回调
         * - {@link onPlayAudioFrame}：混音前的每一路远程用户的音频数据
         * - {@link onMixedPlayAudioFrame}：将各路音频混合之后并最终要由系统播放出的音频数据回调
         *
         * @note 设置回调为空即代表停止自定义音频回调，反之，设置回调不为空则代表启动自定义音频回调。
         */
        public abstract int setAudioFrameCallback(ITRTCAudioFrameCallback callback);

        /**
         * 10.13 设置本地麦克风采集出的音频帧回调格式。
         *
         * 本接口用于设置 {@link onCapturedAudioFrame} 回调出来的 AudioFrame 的格式：
         * - sampleRate：采样率，支持：16000、32000、44100、48000。
         * - channel：声道数（如果是立体声，数据是交叉的），单声道：1； 双声道：2。
         * - samplesPerCall：采样点数，定义回调数据帧长。帧长必须为 10ms 的整数倍。
         * 如果希望用毫秒数计算回调帧长，则将毫秒数转换成采样点数的公式为：`采样点数 = 毫秒数 * 采样率 / 1000`。
         * 举例：48000Hz 采样率希望回调 20ms 帧长的数据，则采样点数应该填：`960 = 20 * 48000 / 1000`。
         * @param format 音频数据回调格式。
         * @return 0：成功；<0：错误。
         * @note
         * 最终回调的帧长度是以字节为单位，采样点数转换成字节数的计算公式为：`字节数 = 采样点数 * channel * 2（位宽）`。举例：48000Hz 采样率，双声道，20ms 帧长，采样点数为 960，字节数为 `3840 = 960 * 2 * 2`。
         */
        public abstract int setCapturedAudioFrameCallbackFormat(TRTCAudioFrameCallbackFormat format);

        /**
         * 10.14 设置经过前处理后的本地音频帧回调格式。
         *
         * 本接口用于设置 {@link onLocalProcessedAudioFrame} 回调出来的 AudioFrame 的格式：
         * - sampleRate：采样率，支持：16000、32000、44100、48000。
         * - channel：声道数（如果是立体声，数据是交叉的），单声道：1； 双声道：2。
         * - samplesPerCall：采样点数，定义回调数据帧长。帧长必须为 10ms 的整数倍。
         * 如果希望用毫秒数计算回调帧长，则将毫秒数转换成采样点数的公式为：`采样点数 = 毫秒数 * 采样率 / 1000`。
         * 举例：48000Hz 采样率希望回调20ms帧长的数据，则采样点数应该填: `960 = 20 * 48000 / 1000`。
         * @param format 音频数据回调格式。
         * @return 0：成功；<0：错误。
         * @note
         * 最终回调的帧长度是以字节为单位，采样点数转换成字节数的计算公式为：`字节数 = 采样点数 * channel * 2（位宽）`。 举例：48000Hz 采样率，双声道，20ms 帧长，采样点数为 960，字节数为 `3840 = 960 * 2 * 2`。
         */
        public abstract int setLocalProcessedAudioFrameCallbackFormat(TRTCAudioFrameCallbackFormat format);

        /**
         * 10.15 设置最终要由系统播放出的音频帧回调格式。
         *
         * 本接口用于设置 {@link onMixedPlayAudioFrame} 回调出来的 AudioFrame 的格式：
         * - sampleRate：采样率，支持：16000、32000、44100、48000。
         * - channel：声道数（如果是立体声，数据是交叉的），单声道：1； 双声道：2。
         * - samplesPerCall：采样点数，定义回调数据帧长。帧长必须为 10ms 的整数倍。
         * 如果希望用毫秒数计算回调帧长，则将毫秒数转换成采样点数的公式为：`采样点数 = 毫秒数 * 采样率 / 1000`。
         * 举例：48000Hz 采样率希望回调 20ms 帧长的数据，则采样点数应该填: `960 = 20 * 48000 / 1000`。
         * @param format 音频数据回调格式。
         * @return 0：成功；<0：错误。
         * @note
         * 最终回调的帧长度是以字节为单位，采样点数转换成字节数的计算公式为：`字节数 = 采样点数 * channel * 2（位宽）`。 举例：48000Hz 采样率，双声道，20ms 帧长，采样点数为 960，字节数为 `3840 = 960 * 2 * 2`。
         */
        public abstract int setMixedPlayAudioFrameCallbackFormat(TRTCAudioFrameCallbackFormat format);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    自定义消息发送接口
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 11.1 使用 UDP 通道发送自定义消息给房间内所有用户。
         *
         * 该接口可以让您借助 TRTC 的 UDP 通道，向当前房间里的其他用户广播自定义数据，以达到传输信令的目的。
         * 房间中的其他用户可以通过 {@link TRTCCloudCallback} 中的 onRecvCustomCmdMsg 回调接收消息。
         * @param cmdID 消息 ID，取值范围为 [1, 10]。
         * @param data 待发送的消息，单个消息的最大长度被限制为 1KB。
         * @param reliable 是否可靠发送，可靠发送可以获得更高的发送成功率，但可靠发送比不可靠发送会带来更大的接收延迟。
         * @param ordered 是否要求有序，即是否要求接收端的数据包顺序和发送端的数据包顺序一致（这会带来一定的接收延时）。
         * @return true：消息已经发出；false：消息发送失败。
         * @note
         * 1. 发送消息到房间内所有用户（暂时不支持 Web/小程序端），每秒最多能发送30条消息（与 {@link sendSEIMsg} 共享限制）。
         * 2. 每个包最大为 1KB，超过则很有可能会被中间路由器或者服务器丢弃。
         * 3. 每个客户端每秒最多能发送总计 16KB 数据（与 {@link sendSEIMsg} 共享限制）。
         * 4. 请将 `reliable` 和 `ordered` 同时设置为 true 或同时设置为 false，暂不支持交叉设置。
         * 5. 强烈建议您将不同类型的消息设定为不同的 cmdID，这样可以在要求有序的情况下减小消息时延。
         * 6. 目前仅支持主播身份。
         */
        public abstract bool sendCustomCmdMsg(int cmdId, byte[] data, int dataSize, bool reliable, bool ordered);

        /**
         * 11.2 使用 SEI 通道发送自定义消息给房间内所有用户。
         *
         * 该接口可以让您借助 TRTC 的 SEI 通道，向当前房间里的其他用户广播自定义数据，已达到传输信令的目的。
         * 视频帧的头部有一个叫做 SEI 的头部数据块，该接口的原理就是利用这个被称为 SEI 的头部数据块，将您要发送的自定义信令嵌入其中，使其同视频帧一并发送出去。
         * 因此，与 {@link sendCustomCmdMsg} 相比，SEI 通道传输的信令具有更好的兼容性：信令可以伴随着视频帧一直传输到直播 CDN 上。
         * 不过，由于视频帧头部的数据块不能太大，建议您使用该接口时，尽量将信令控制在几个字节的大小。
         * 最常见的用法是把自定义的时间戳（timestamp）用本接口嵌入视频帧中，实现消息和画面的完美对齐（比如：教育场景下的课件和视频信号的对齐）。
         * 房间中的其他用户可以通过 {@link TRTCCloudCallback} 中的 {@link onRecvSEIMsg} 回调接收消息。
         * @param data 待发送的数据，最大支持 1KB（1000字节）的数据大小
         * @param repeatCount 发送数据次数
         * @return true：消息已通过限制，等待后续视频帧发送；false：消息被限制发送
         * @note 本接口有以下限制：
         * 1. 数据在接口调用完后不会被即时发送出去，而是从下一帧视频帧开始与视频帧一起发送。
         * 2. 发送消息到房间内所有用户，每秒最多能发送 40 条消息（与 {@link sendCustomCmdMsg} 共享限制）。
         * 3. 每个包最大为 1KB，若发送大量数据，会导致视频码率增大，可能导致视频画质下降甚至卡顿（与 {@link sendCustomCmdMsg} 共享限制）。
         * 4. 每个客户端每秒最多能发送总计 16KB 数据（与 {@link sendCustomCmdMsg} 共享限制）。
         * 5. 若指定多次发送（repeatCount > 1），则数据会被带在后续的连续 repeatCount 个视频帧中发送出去，同样会导致视频码率增大。接收消息 {@link onRecvSEIMsg} 回调也可能会收到多次相同的消息，需要去重。
         */
        public abstract bool sendSEIMsg(byte[] data, int dataSize, int repeatCount);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    网络测试接口
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 12.1 开始进行网速测试（进入房间前使用）。
         *
         * @param params 测速选项
         * @return 接口调用结果，< 0：失败
         * @note
         * 1. 测速过程将产生少量的基础服务费用，详见 [计费概述 > 基础服务](https://cloud.tencent.com/document/product/647/17157#.E5.9F.BA.E7.A1.80.E6.9C.8D.E5.8A.A1) 文档说明。
         * 2. 请在进入房间前进行网速测试，在房间中网速测试会影响正常的音视频传输效果，而且由于干扰过多，网速测试结果也不准确。
         * 3. 同一时间只允许一项网速测试任务运行。
         */
        public abstract void startSpeedTest(TRTCSpeedTestParams testParams);

        /**
         * 12.2 停止网络测速。
         */
        public abstract void stopSpeedTest();

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    调试相关接口
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 13.0 获取 unity SDK 版本信息。
         */
        public abstract string getScriptVersion();

        /**
         * 13.1 获取 SDK 版本信息。
         */
        public abstract string getSDKVersion();

        /**
         * 13.2 设置 Log 输出级别。
         *
         * @param level 参见 {@link TRTCLogLevel}，默认值：{@link TRTCLogLevelNone}
         */
        public abstract void setLogLevel(TRTCLogLevel level);

        /**
         * 13.3 启用/禁用控制台日志打印。
         *
         * @param enabled 指定是否启用，默认：禁止状态。
         */
        public abstract void setConsoleEnabled(bool enabled);

        /**
         * 13.4 启用/禁用日志的本地压缩。
         *
         * 开启压缩后，Log 存储体积明显减小，但需要腾讯云提供的 Python 脚本解压后才能阅读。
         * 禁用压缩后，Log 采用明文存储，可以直接用记事本打开阅读，但占用空间较大。
         * @param enabled 指定是否启用，默认为启动状态
         */
        public abstract void setLogCompressEnabled(bool enabled);

        /**
         * 13.5 设置本地日志的保存路径。
         *
         * 通过该接口您可以更改 SDK 本地日志的默认存储路径，SDK 默认的本地日志的存储位置：
         * - Windows 平台：在 C:/Users/[系统用户名]/AppData/Roaming/liteav/log，即 %appdata%/liteav/log 下。
         * - iOS 或 Mac 平台：在 sandbox Documents/log 下。
         * - Android 平台：在 /app私有目录/files/log/liteav/ 下。
         *
         * @note 请务必在所有其他接口之前调用，并且保证您指定的目录是存在的，并且您的应用程序拥有对该目录的读写权限。
         * @param path 存储日志的路径
         */
        public abstract void setLogDirPath(string path);

        /**
         * 13.6 设置日志回调。
         */
        public abstract void setLogCallback(ITRTCLogCallback callback);

        /**
         * 13.9 调用实验性接口。
         */
        public abstract void callExperimentalAPI(string jsonStr);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    加密接口
        //
        /////////////////////////////////////////////////////////////////////////////////
    }
}
