/**
 * Copyright (c) 2021 Tencent. All rights reserved.
 * Module:   TRTCCloudCallback @ TXLiteAVSDK
 * Function: 腾讯云实时音视频的事件回调接口
 */
using System;
using System.Runtime.InteropServices;
namespace trtc {

    public interface ITRTCCloudCallback {
        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    错误和警告事件
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 1.1 错误事件回调。
         *
         * 错误事件，表示 SDK 抛出的不可恢复的错误，比如进入房间失败或设备开启失败等。
         * 参考文档：[错误码表](https://cloud.tencent.com/document/product/647/38308)
         * @param errCode 错误码
         * @param errMsg  错误信息
         * @param extInfo 扩展信息字段，个别错误码可能会带额外的信息帮助定位问题
         */
        void onError(TXLiteAVError errCode, String errMsg, IntPtr arg);

        /**
         * 1.2 警告事件回调。
         *
         * 警告事件，表示 SDK 抛出的提示性问题，比如视频出现卡顿或 CPU 使用率太高等。
         * 参考文档：[错误码表](https://cloud.tencent.com/document/product/647/38308)
         * @param warningCode 警告码
         * @param warningMsg 警告信息
         * @param extInfo 扩展信息字段，个别警告码可能会带额外的信息帮助定位问题
         */
        void onWarning(TXLiteAVWarning warningCode, String warningMsg, IntPtr arg);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    房间相关事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 2.1 进入房间成功与否的事件回调。
         *
         * 调用 TRTCCloud 中的 {@link enterRoom} 接口执行进房操作后，会收到来自 {@link TRTCCloudCallback} 的 `onEnterRoom(result)` 回调：
         * - 如果加入成功，回调 result 会是一个正数（result > 0），代表进入房间所消耗的时间，单位是毫秒（ms）。
         * - 如果加入失败，回调 result 会是一个负数（result < 0），代表失败原因的错误码。
         *  进房失败的错误码含义请参见[错误码表](https://cloud.tencent.com/document/product/647/38308)。
         * @param result result > 0 时为进房耗时（ms），result < 0 时为进房错误码。
         * @note
         * 1. 在 Ver6.6 之前的版本，只有进房成功会抛出 `onEnterRoom(result)` 回调，进房失败由 {@link onError} 回调抛出。
         * 2. 在 Ver6.6 之后的版本：无论进房成功或失败，均会抛出 `onEnterRoom(result)` 回调，同时进房失败也会有 {@link onError} 回调抛出。
         */
        void onEnterRoom(int result);

        /**
         * 2.2 离开房间的事件回调。
         *
         * 调用 TRTCCloud 中的 {@link exitRoom} 接口会执行退出房间的相关逻辑，例如释放音视频设备资源和编解码器资源等。
         * 待 SDK 占用的所有资源释放完毕后，SDK 会抛出 `onExitRoom()` 回调通知到您。
         * 如果您要再次调用 {@link enterRoom} 或者切换到其他的音视频 SDK，请等待 `onExitRoom` 回调到来后再执行相关操作。否则可能会遇到例如摄像头、麦克风设备被强占等各种异常问题。
         * @param reason 离开房间原因，0：主动调用 exitRoom 退出房间；1：被服务器踢出当前房间；2：当前房间整个被解散 3: 服务器状态异常。
         */
        void onExitRoom(int reason);

        /**
         * 2.3 切换角色的事件回调。
         *
         * 调用 TRTCCloud 中的 {@link switchRole} 接口会切换主播和观众的角色，该操作会伴随一个线路切换的过程，待 SDK 切换完成后，会抛出 `onSwitchRole()` 事件回调。
         * @param errCode 错误码，ERR_NULL 代表切换成功，其他请参见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param errMsg  错误信息。
         */
        void onSwitchRole(TXLiteAVError errCode, String errMsg);

        /**
         * 2.4 切换房间的结果回调。
         *
         * 调用 TRTCCloud 中的 {@link switchRoom} 接口可以让用户快速地从一个房间切换到另一个房间，待 SDK 切换完成后，会抛出 `onSwitchRoom()` 事件回调。
         * @param errCode 错误码，ERR_NULL 代表切换成功，其他请参见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param errMsg  错误信息。
         */
        void onSwitchRoom(TXLiteAVError errCode, string errMsg);

        /**
         * 2.5 请求跨房通话的结果回调。
         *
         * 调用 TRTCCloud 中的 {@link connectOtherRoom} 接口会将两个不同房间中的主播拉通视频通话，也就是所谓的“主播PK”功能。
         * 调用者会收到 `onConnectOtherRoom()` 回调来获知跨房通话是否成功。
         * @param userId  要跨房通话的另一个房间中的主播的用户 ID。
         * @param errCode 错误码，ERR_NULL 代表切换成功，其他请参见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param errMsg  错误信息。
         */
        void onConnectOtherRoom(string userId, TXLiteAVError errCode, string errMsg);

        /**
         * 2.6 结束跨房通话的结果回调。
         *
         * 调用 TRTCCloud 中的 {@link disConnectOtherRoom} 接口会结束两个不同房间中的主播视频通话，也就是所谓的“主播PK”功能。
         * 调用者会收到 `onDisconnectOtherRoom()` 回调来获知结束跨房通话是否成功。
         * @param errCode 错误码，ERR_NULL 代表成功，其他请参见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param errMsg  错误信息。
         */
        void onDisconnectOtherRoom(TXLiteAVError errCode, string errMsg);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    用户相关事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 3.1 有用户加入当前房间。
         *
         * 出于性能方面的考虑，在 TRTC 两种不同的应用场景（即 `AppScene`，在 {@link enterRoom} 时通过第二个参数指定）下，该通知的行为会有差别：
         * - 直播类场景（{@link TRTCAppSceneLIVE} 和 {@link TRTCAppSceneVoiceChatRoom}）：该场景下的用户区分主播和观众两种角色，只有主播进入房间时才会触发该通知，观众进入房间时不会触发该通知。
         * - 通话类场景（{@link TRTCAppSceneVideoCall} 和 {@link TRTCAppSceneAudioCall}）：该场景下的用户没有角色的区分（可认为都是主播），任何用户进入房间都会触发该通知。
         * @param userId 远端用户的用户标识。
         * @note
         * 1. 事件回调 `onRemoteUserEnterRoom` 和 {@link onRemoteUserLeaveRoom} 只适用于维护当前房间里的“用户列表”，有此事件回调不代表一定有视频画面。
         * 2. 如果需要显示远程画面，请监听代表某个用户是否有视频画面的 {@link onUserVideoAvailable} 事件回调。
         */
        void onRemoteUserEnterRoom(String userId);

        /**
         * 3.2 有用户离开当前房间。
         *
         * 与 {@link onRemoteUserEnterRoom} 相对应，在两种不同的应用场景（即 AppScene，在 {@link enterRoom} 时通过第二个参数指定）下，该通知的行为会有差别：
         * - 直播类场景（{@link TRTCAppSceneLIVE} 和 {@link TRTCAppSceneVoiceChatRoom}）：该场景下的用户区分主播和观众两种角色，只有主播离开房间时才会触发该通知，观众离开房间时不会触发该通知。
         * - 通话类场景（{@link TRTCAppSceneVideoCall} 和 {@link TRTCAppSceneAudioCall}）：该场景下的用户没有角色的区分（可认为都是主播），任何用户离开房间都会触发该通知。
         * @param userId 远端用户的用户标识。
         * @param reason 离开原因，0表示用户主动退出房间，1表示用户超时退出，2表示被踢出房间，3表示主播因切换到观众退出房间。
         */
        void onRemoteUserLeaveRoom(String userId, int reason);

        /**
         * 3.3 某远端用户发布/取消了主路视频画面。
         *
         * **主路画面** 一般被用于承载摄像头画面。当您收到 `onUserVideoAvailable(userId, true)` 通知时，表示该路画面已经有可播放的视频帧到达。
         * 此时，您需要调用 {@link startRemoteView} 接口订阅该用户的远程画面，订阅成功后，您会继续收到该用户的首帧画面渲染回调 `onFirstVideoFrame(userId)`。
         * 当您收到 `onUserVideoAvailable(userId, false)` 通知时，表示该路远程画面已经被关闭，关闭的原因可能是该用户调用了 {@link muteLocalVideo} 或 {@link stopLocalPreview}。
         * @param userId 远端用户的用户标识。
         * @param available 该用户是否发布（或取消发布）了主路视频画面，true：发布；false：取消发布。
         */
        void onUserVideoAvailable(String userId, bool available);

        /**
         * 3.4 某远端用户发布/取消了辅路视频画面。
         *
         * “辅路画面”一般被用于承载屏幕分享的画面。当您收到 `onUserSubStreamAvailable(userId, true)` 通知时，表示该路画面已经有可播放的视频帧到达。
         * 此时，您需要调用 {@link startRemoteView} 接口订阅该用户的远程画面，订阅成功后，您会继续收到该用户的首帧画面渲染回调 `onFirstVideoFrame(userId)`。
         * @param userId 远端用户的用户标识。
         * @param available 该用户是否发布（或取消发布）了辅路视频画面，true: 发布；false：取消发布。
         * @note 显示辅路画面使用的函数是 {@link startRemoteView} 而非 {@link startRemoteSubStreamView}，{@link startRemoteSubStreamView} 已经被废弃。
         */
        void onUserSubStreamAvailable(String userId, bool available);

        /**
         * 3.5 某远端用户发布/取消了自己的音频。
         *
         * 当您收到 `onUserAudioAvailable(userId, true)` 通知时，表示该用户发布了自己的声音，此时 SDK 的表现为：
         * - 在自动订阅模式下，您无需做任何操作，SDK 会自动播放该用户的声音。
         * - 在手动订阅模式下，您可以通过 {@link muteRemoteAudio}(userid, false) 来播放该用户的声音。
         * @param userId 远端用户的用户标识。
         * @param available 该用户是否发布（或取消发布）了自己的音频，true: 发布；false：取消发布。
         * @note SDK 默认使用自动订阅模式，您可以通过 {@link setDefaultStreamRecvMode} 设置为手动订阅，但需要在您进入房间之前调用才生效。
         */
        void onUserAudioAvailable(String userId, bool available);

        /**
         * 3.6 SDK 开始渲染自己本地或远端用户的首帧画面。
         *
         * SDK 会在渲染自己本地或远端用户的首帧画面时抛出该事件，您可以通过回调事件中的 userId 参数来判断事件来自于“本地”还是来自于“远端”。
         * - 如果 userId 为空值，代表 SDK 已经开始渲染自己本地的视频画面，不过前提是您已经调用了 {@link startLocalPreview} 或 {@link startScreenCapture}。
         * - 如果 userId 不为空，代表 SDK 已经开始渲染远端用户的视频画面，不过前提是您已经调用了 {@link startRemoteView} 订阅了该用户的视频画面。
         * @param userId 本地或远端的用户标识，如果 userId 为空值代表自己本地的首帧画面已到来，userId 不为空则代表远端用户的首帧画面已到来。
         * @param streamType 视频流类型：主路（Main）一般用于承载摄像头画面，辅路（Sub）一般用于承载屏幕分享画面。
         * @param width  画面的宽度。
         * @param height 画面的高度。
         * @note
         * 1. 只有当您调用了 {@link startLocalPreview} 或 {@link startScreenCapture} 之后，才会触发自己本地的首帧画面事件回调。
         * 2. 只有当您调用了 {@link startRemoteView} 或 {@link startRemoteSubStreamView} 之后，才会触发远端用户的首帧画面事件回调。
         */
        void onFirstVideoFrame(String userId, TRTCVideoStreamType streamType, int width, int height);

        /**
         * 3.7 SDK 开始播放远端用户的首帧音频。
         *
         * SDK 会在播放远端用户的首帧音频时抛出该事件，本地音频的首帧事件暂不抛出。
         * @param userId 远端用户的用户标识。
         */
        void onFirstAudioFrame(String userId);

        /**
         * 3.8 自己本地的首个视频帧已被发布出去。
         *
         * 当您成功进入房间并通过 {@link startLocalPreview} 或 {@link startScreenCapture} 开启本地视频采集之后（开启采集和进入房间的先后顺序无影响），SDK 就会开始进行视频编码，并通过自身的网络模块向云端发布自己本地的视频数据。当 SDK
         * 成功地向云端送出自己的第一帧视频数据帧以后，就会抛出 `onSendFirstLocalVideoFrame` 事件回调。
         * @param streamType 视频流类型：主路（Main）一般用于承载摄像头画面，辅路（Sub）一般用于承载屏幕分享画面。
         */
        void onSendFirstLocalVideoFrame(TRTCVideoStreamType streamType);

        /**
         * 3.9 自己本地的首个音频帧已被发布出去。
         *
         * 当您成功进入房间并通过 {@link startLocalAudio} 开启本地音频采集之后（开启采集和进入房间的先后顺序无影响），SDK 就会开始进行音频编码，并通过自身的网络模块向云端发布自己本地的音频数据。当 SDK
         * 成功地向云端送出自己的第一帧音频数据帧以后，就会抛出 `onSendFirstLocalAudioFrame` 事件回调。
         */
        void onSendFirstLocalAudioFrame();

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    网络和技术指标统计回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 4.1 网络质量的实时统计回调。
         *
         * 该统计回调每间隔2秒抛出一次，用于通知 SDK 感知到的当前网络的上行和下行质量。
         * SDK 会使用一组内嵌的自研算法对当前网络的延迟高低、带宽大小以及稳定情况进行评估，并计算出一个评估结果：如果评估结果为 1（Excellent） 代表当前的网络情况非常好，如果评估结果为 6（Down）代表当前网络无法支撑 TRTC 的正常通话。
         * @param localQuality 上行网络质量。
         * @param remoteQuality 下行网络质量，代表数据流历经“远端->云端->本端”这样一条完整的传输链路后，最终在本端统计到的数据质量。因此，这里的下行网络质量代表的是远端上行链路与本端下行链路共同影响的结果。
         * @note 暂时无法通过该接口单独判定远端用户的上行质量。
         */
        void onNetworkQuality(TRTCQualityInfo localQuality, TRTCQualityInfo[] remoteQuality, UInt32 remoteQualityCount);

        /**
         * 4.2 音视频技术指标的实时统计回调。
         *
         * 该统计回调每间隔2秒抛出一次，用于通知 SDK 内部音频、视频以及网络相关的专业技术指标，这些信息在 {@link TRTCStatistics} 均有罗列：
         * - 视频统计信息：视频的分辨率（resolution）、帧率（FPS）和比特率（bitrate）等信息。
         * - 音频统计信息：音频的采样率（samplerate）、声道（channel）和比特率（bitrate）等信息。
         * - 网络统计信息：SDK 和云端一次往返（SDK > Cloud > SDK）的网络耗时（rtt）、丢包率（loss）、上行流量（sentBytes）和下行流量（receivedBytes）等信息。
         * @param statistics 统计数据，包括自己本地的统计信息和远端用户的统计信息，详情请参考 {@link TRTCStatistics}。
         * @note 如果您只需要获知当前网络质量的好坏，并不需要花太多时间研究本统计回调，更推荐您使用 {@link onNetworkQuality} 来解决问题。
         */
        void onStatistics(TRTCStatistics statis);

        /**
         * 4.3 网速测试的结果回调。
         *
         * 该统计回调由 {@link startSpeedTest} 触发。
         * @param result 网速测试结果，包括丢包、往返延迟、上下行的带宽速率，详情请参见 {@link TRTCSpeedTestResult}。
         */
        void onSpeedTestResult(TRTCSpeedTestResult result);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    与云端连接情况的事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 5.1 SDK 与云端的连接已经断开。
         *
         * SDK 会在跟云端的连接断开时抛出此事件回调，导致断开的原因大多是网络不可用或者网络切换所致，比如用户在通话中走进电梯时就可能会遇到此事件。
         * 在抛出此事件之后，SDK 会努力跟云端重新建立连接，重连过程中会抛出 {@link onTryToReconnect}，连接恢复后会抛出 {@link onConnectionRecovery} 。
         * 所以，SDK 会在如下三个连接相关的事件中按如下规律切换：
         * ![](https://qcloudimg.tencent-cloud.cn/raw/fb3c40a4fca55b0010d385cf3b2472cd.png)
         */
        void onConnectionLost();

        /**
         * 5.2 SDK 正在尝试重新连接到云端。
         *
         * SDK 会在跟云端的连接断开时抛出 {@link onConnectionLost}，之后会努力跟云端重新建立连接并抛出本事件，连接恢复后会抛出 {@link onConnectionRecovery}。
         */
        void onTryToReconnect();

        /**
         * 5.3 SDK 与云端的连接已经恢复。
         *
         * SDK 会在跟云端的连接断开时抛出 {@link onConnectionLost}，之后会努力跟云端重新建立连接并抛出 {@link onTryToReconnect}，连接恢复后会抛出本事件回调。
         */
        void onConnectionRecovery();

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    硬件设备相关事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 6.1 摄像头准备就绪。
         *
         * 当您调用 {@link startLocalPreview} 之后，SDK 会尝试启动摄像头，如果摄像头能够启动成功就会抛出本事件。
         * 如果启动失败，大概率是因为当前应用没有获得访问摄像头的权限，或者摄像头当前正在被其他程序独占使用中。
         * 您可以通过捕获 {@link onError} 事件回调获知这些异常情况并通过 UI 界面提示用户。
         */
        void onCameraDidReady();

        /**
         * 6.2 麦克风准备就绪。
         *
         * 当您调用 {@link startLocalAudio} 之后，SDK 会尝试启动麦克风，如果麦克风能够启动成功就会抛出本事件。
         * 如果启动失败，大概率是因为当前应用没有获得访问麦克风的权限，或者麦克风当前正在被其他程序独占使用中。
         * 您可以通过捕获 {@link onError} 事件回调获知这些异常情况并通过 UI 界面提示用户。
         */
        void onMicDidReady();

        /**
         * 6.3 当前音频路由发生变化（仅适用于移动设备）。
         *
         * 所谓“音频路由”，是指声音是从手机的扬声器还是从听筒中播放出来，音频路由变化也就是声音的播放位置发生了变化。
         * - 当音频路由为听筒时，声音比较小，只有将耳朵凑近才能听清楚，隐私性较好，适合用于接听电话。
         * - 当音频路由为扬声器时，声音比较大，不用将手机贴脸也能听清，因此可以实现“免提”的功能。
         * - 当音频路由为有线耳机。
         * - 当音频路由为蓝牙耳机。
         * - 当音频路由为USB专业声卡设备。
         * @param route 音频路由，即声音由哪里输出（扬声器、听筒）。
         * @param fromRoute 变更前的音频路由。
         */
        void onAudioRouteChanged(TRTCAudioRoute newRoute, TRTCAudioRoute oldRoute);

        /**
         * 6.4 音量大小的反馈回调。
         *
         * SDK 可以评估每一路音频的音量大小，并每隔一段时间抛出该事件回调，您可以根据音量大小在 UI 上做出相应的提示，比如`波形图`或`音量槽`。
         * 要完成这个功能， 您需要先调用 {@link enableAudioVolumeEvaluation} 开启这个能力并设定事件抛出的时间间隔。
         * 需要补充说明的是，无论当前房间中是否有人说话，SDK 都会按照您设定的时间间隔定时抛出此事件回调。
         * @param userVolumes 是一个数组，用于承载所有正在说话的用户的音量大小，取值范围 [0, 100]。
         * @param totalVolume 所有远端用户的总音量大小，取值范围 [0, 100]。
         * @note userVolumes 为一个数组，对于数组中的每一个元素，当 userId 为空时表示本地麦克风采集的音量大小，当 userId 不为空时代表远端用户的音量大小。
         */
        void onUserVoiceVolume(TRTCVolumeInfo[] userVolumes, UInt32 userVolumesCount, UInt32 totalVolume);

        /**
         * 6.5 本地设备的通断状态发生变化（仅适用于桌面系统）。
         *
         * 当本地设备（包括摄像头、麦克风以及扬声器）被插入或者拔出时，SDK 便会抛出此事件回调。
         * @param deviceId 设备 ID。
         * @param deviceType 设备类型。0：麦克风设备；1：扬声器设备；2：摄像头设备。
         * @param state 通断状态，0：设备已添加；1：设备已被移除；2：设备已启用。
         */
        void onDeviceChange(String deviceId, TRTCDeviceType type, TRTCDeviceState state);

        /**
         * 6.6 当前麦克风的系统采集音量发生变化。
         *
         * 在 Mac 或 Windows 这样的桌面操作系统上，用户可以在设置中心找到声音相关的设置面板，并设置麦克风的采集音量大小。
         * 用户将麦克风的采集音量设置得越大，麦克风采集到的声音的原始音量也就会越大，反之就会越小。
         * 在有些型号的键盘以及笔记本电脑上，用户还可以通过按下`禁用麦克风`按钮（图标是一个`话筒`叠加了一道代表禁用的斜线）来将麦克风静音。
         * 当用户通过系统设置界面或者通过键盘上的快捷键设定操作系统的麦克风采集音量时，SDK 便会抛出此事件。
         * @param volume 系统采集音量，取值范围 0 - 100，用户可以在系统的声音设置面板上进行拖拽调整。
         * @param muted 麦克风是否被用户禁用了：true 被禁用，false 被启用。
         * @note 您需要调用 {@link enableAudioVolumeEvaluation} 接口并设定 interval 大于 0 开启次事件回调，设定 interval 等于 0 关闭此事件回调。
         */
        void onAudioDeviceCaptureVolumeChanged(int volume, bool muted);

        /**
         * 6.7 当前系统的播放音量发生变化。
         *
         * 在 Mac 或 Windows 这样的桌面操作系统上，用户可以在设置中心找到声音相关的设置面板，并设置系统的播放音量大小。
         * 在有些型号的键盘以及笔记本电脑上，用户还可以通过按下“静音”按钮（图标是一个喇叭上叠加了一道代表禁用的斜线）来将系统静音。
         * 当用户通过系统设置界面或者通过键盘上的快捷键设定操作系统的播放音量时，SDK 便会抛出此事件。
         * @param volume 系统播放音量，取值范围 [0, 100]，用户可以在系统的声音设置面板上进行拖拽调整。
         * @param muted 系统是否被用户静音了：true 被静音，false 已恢复。
         * @note 您需要调用 {@link enableAudioVolumeEvaluation} 接口并设定 interval 大于 0 开启次事件回调，设定 interval 等于 0 关闭此事件回调。
         */
        void onAudioDevicePlayoutVolumeChanged(int volume, bool muted);

        /**
         * 6.9 测试麦克风时的音量回调。
         *
         * 当您调用 {@link startMicDeviceTest} 测试麦克风是否正常工作时，SDK 会不断地抛出此回调，参数中的 volume 代表当前麦克风采集到的音量大小。
         * 如果在测试期间 volume 出现了大小波动的情况，说明麦克风状态健康；如果 volume 的数值始终是 0，说明麦克风的状态异常，需要提示用户进行更换。
         * @param volume 麦克风采集到的音量值，取值范围 [0, 100]。
         */
        void onTestMicVolume(int volume);

        /**
         * 6.10 测试扬声器时的音量回调。
         *
         * 当您调用 {@link startSpeakerDeviceTest} 测试扬声器是否正常工作时，SDK 会不断地抛出此回调。
         * 参数中的 volume 代表的是 SDK 提交给系统扬声器去播放的声音的音量值大小，如果该数值持续变化，但用户反馈听不到声音，则说明扬声器状态异常。
         * @param volume SDK 提交给扬声器去播放的声音的音量，取值范围 [0, 100]。
         */
        void onTestSpeakerVolume(int volume);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    自定义消息的接收事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 7.1 收到自定义消息的事件回调。
         *
         * 当房间中的某个用户使用 {@link sendCustomCmdMsg} 发送自定义 UDP 消息时，房间中的其他用户可以通过 `onRecvCustomCmdMsg` 事件回调接收到该条消息。
         * @param userId 用户标识。
         * @param cmdID 命令 ID。
         * @param seq   消息序号。
         * @param message 消息数据。
         */
        void onRecvCustomCmdMsg(String userId, int cmdID, int seq, byte[] message, int messageSize);

        /**
         * 7.2 自定义消息丢失的事件回调。
         *
         * 当您使用 {@link sendCustomCmdMsg} 发送自定义 UDP 消息时，即使设置了可靠传输（reliable），也无法确 100% 不丢失，只是丢消息概率极低，能满足常规可靠性要求。
         * 在发送端设置了可靠运输（reliable）后，SDK 都会通过此回调通知过去时间段内（通常为5s）传输途中丢失的自定义消息数量统计信息。
         * @param userId 用户标识。
         * @param cmdID 命令 ID。
         * @param errCode 错误码。
         * @param missed 丢失的消息数量。
         * @note  只有在发送端设置了可靠传输（reliable），接收方才能收到消息的丢失回调。
         */
        void onMissCustomCmdMsg(String userId, int cmdID, int errCode, int missed);

        /**
         * 7.3 收到 SEI 消息的回调。
         *
         * 当房间中的某个用户使用 {@link sendSEIMsg} 借助视频数据帧发送 SEI 消息时，房间中的其他用户可以通过 `onRecvSEIMsg` 事件回调接收到该条消息。
         * @param userId   用户标识。
         * @param message  数据。
         */
        void onRecvSEIMsg(String userId, Byte[] message, UInt32 msgSize);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    CDN 相关事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 8.1 开始向腾讯云直播 CDN 上发布音视频流的事件回调。
         *
         * 当您调用 {@link startPublishing} 开始向腾讯云直播 CDN 上发布音视频流时，SDK 会立刻将这一指令同步给云端服务器。
         * 随后 SDK 会收到来自云端服务器的处理结果，并将指令的执行结果通过本事件回调通知给您。
         * @param err 0表示成功，其余值表示失败，详见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param errMsg 具体错误原因。
         */
        void onStartPublishing(int errCode, String errMsg);

        /**
         * 8.2 停止向腾讯云直播 CDN 上发布音视频流的事件回调。
         *
         * 当您调用 {@link stopPublishing} 停止向腾讯云直播 CDN 上发布音视频流时，SDK 会立刻将这一指令同步给云端服务器。
         * 随后 SDK 会收到来自云端服务器的处理结果，并将指令的执行结果通过本事件回调通知给您。
         * @param err 0表示成功，其余值表示失败，详见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param errMsg 具体错误原因。
         */
        void onStopPublishing(int errCode, String errMsg);

        /**
         * 8.5 设置云端混流的排版布局和转码参数的事件回调。
         *
         * 当您调用 {@link setMixTranscodingConfig} 调整云端混流的排版布局和转码参数时，SDK 会立刻将这一调整指令同步给云端服务器。随后 SDK 会收到来自云端服务器的处理结果，并将指令的执行结果通过本事件回调通知给您。
         * @param err 错误码：0表示成功，其余值表示失败，详见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param errMsg 具体的错误原因。
         */
        void onSetMixTranscodingConfig(int errCode, String errMsg);

        /**
         * 8.6 开始发布媒体流的事件回调。
         *
         * 当您调用 {@link startPublishMediaStream} 开始向 TRTC 后台服务发布媒体流时，SDK 会立刻将这一指令同步给云端服务器，随后 SDK 会收到来自云端服务器的处理结果，并将指令的执行结果通过本事件回调通知给您。
         * @param taskId 当请求成功时，TRTC 后台会在回调中提供给您这项任务的 taskId，后续您可以通过该 taskId 结合 {@link updatePublishMediaStream} 和 {@link stopPublishMediaStream} 进行更新和停止。
         * @param code 回调结果，0 表示成功，其余值表示失败，详见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param message 具体回调信息。
         * @param extraInfo 扩展信息字段，个别错误码可能会带额外的信息帮助定位问题。
         * 当 code 返回值为 ERR_SERVER_PROCESS_FAILED 时，意味着服务器无法处理您的请求，此时可使用"error_code"做为 key 获取到服务器返回的错误码，具体的错误码以及建议处理措施说明如下：
         * - 2000: 参数错误，建议检查请求参数
         * - 2001: 未开通直播服务，建议去直播控制台开通直播服务
         * - 2021: 未开通转推第三方服务，建议联系产品开通转推第三方服务
         * - 3000: 服务器内部错误，建议重试
         * - 4006: 同一个任务并发冲突，上一个 startPublishMediaStream 任务服务器还在处理中，建议延迟重试
         * - 4007: 同一个任务已经通过 startPublishMediaStream 启动成功，建议使用返回的 taskId 调用 updatePublishMediaStream 接口更新任务
         * - 5001: 资源受限，建议延迟重试
         */
        void onStartPublishMediaStream(string taskID, int code, string message, string extraInfo);

        /**
         * 8.7 更新媒体流的事件回调。
         *
         * 当您调用媒体流发布接口 ({@link updatePublishMediaStream}) 开始向 TRTC 后台服务更新媒体流时，SDK 会立刻将这一指令同步给云端服务器，随后 SDK 会收到来自云端服务器的处理结果，并将指令的执行结果通过本事件回调通知给您。
         * @param taskId 您调用媒体流发布接口 ({@link updatePublishMediaStream}) 时传入的 taskId，会通过此回调再带回给您，用于标识该回调属于哪一次更新请求。
         * @param code 回调结果，0 表示成功，其余值表示失败，详见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param message 具体回调信息。
         * @param extraInfo 扩展信息字段，个别错误码可能会带额外的信息帮助定位问题。
         * 当 code 返回值为 ERR_SERVER_PROCESS_FAILED 时，意味着服务器无法处理您的请求，此时可使用"error_code"做为 key 获取到服务器返回的错误码，具体的错误码以及建议处理措施说明如下：
         * - 2000: 参数错误，建议检查请求参数
         * - 2001: 未开通直播服务，建议去直播控制台开通直播服务
         * - 2002: 找不到任务，建议调用 startPublishMediaStream 接口重新启动任务
         * - 2018: 并发请求乱序，建议使用最新混流参数重试
         * - 2021: 未开通转推第三方服务，建议联系产品开通转推第三方服务
         * - 3000: 服务器内部错误，建议重试
         * - 4003: 任务正在退出中，建议先调用 stopPublishMediaStream 接口停止任务，再调用 startPublishMediaStream 接口启动任务
         */
        void onUpdatePublishMediaStream(string taskID, int code, string message, string extraInfo);

        /**
         * 8.8 停止媒体流的事件回调。
         *
         * 当您调用停止发布媒体流 ({@link stopPublishMediaStream}) 开始向 TRTC 后台服务停止媒体流时，SDK 会立刻将这一指令同步给云端服务器，随后 SDK 会收到来自云端服务器的处理结果，并将指令的执行结果通过本事件回调通知给您。
         * @param taskId 您调用停止发布媒体流 ({@link stopPublishMediaStream}) 时传入的 taskId，会通过此回调再带回给您，用于标识该回调属于哪一次停止请求。
         * @param code 回调结果，0 表示成功，其余值表示失败，详见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param message 具体回调信息。
         * @param extraInfo 扩展信息字段，个别错误码可能会带额外的信息帮助定位问题。
         * 当 code 返回值为 ERR_SERVER_PROCESS_FAILED 时，意味着服务器无法处理您的请求，此时可使用"error_code"做为 key 获取到服务器返回的错误码，具体的错误码以及建议处理措施说明如下：
         * - 2000: 参数错误，建议检查请求参数
         * - 2002: 找不到任务，无需处理
         * - 3000: 服务器内部错误，建议重试
         * - 4003: 任务正在退出中，无需处理
         */
        void onStopPublishMediaStream(string taskID, int code, string message, string extraInfo);

        /**
         * 8.9 RTMP/RTMPS 推流状态发生改变回调。
         *
         * 当您调用 {@link startPublishMediaStream} 开始向 TRTC 后台服务发布媒体流时，SDK 会立刻将这一指令同步给云端服务。
         * 若您在目标推流配置 ({@link TRTCPublishTarget}) 设置了向腾讯或者第三方 CDN 上发布音视频流的 URL 配置，则具体 RTMP 或者 RTMPS 推流状态将通过此回调同步给您。
         * @param cdnUrl 您调用 {@link startPublishMediaStream} 时通过目标推流配置 ({@link TRTCPublishTarget}) 传入的 url，在推流状态变更时，会通过此回调同步给您。
         * @param status 推流状态。
         * - 0：推流未开始或者已结束。在您调用 {@link stopPublishMediaStream} 时会返回该状态。
         * - 1：正在连接 TRTC 服务器和 CDN 服务器。若无法立刻成功，TRTC 后台服务会多次重试尝试推流，并返回该状态（5s回调一次）。如成功进行推流，则进入状态 2；如服务器出错或 60 秒内未成功推流，则进入状态 4。
         * - 2：CDN 推流正在进行。在成功推流后，会返回该状态。
         * - 3：TRTC 服务器和 CDN 服务器推流中断，正在恢复。当 CDN 出现异常，或推流短暂中断时，TRTC 后台服务会自动尝试恢复推流，并返回该状态（5s回调一次）。如成功恢复推流，则进入状态 2；如服务器出错或 60 秒内未成功恢复，则进入状态 4。
         * - 4：TRTC 服务器和 CDN 服务器推流中断，且恢复或连接超时。即此时推流失败，你可以再次调用 {@link updatePublishMediaStream} 尝试推流。
         * - 5：正在断开 TRTC 服务器和 CDN 服务器。在您调用 {@link stopPublishMediaStream} 时，TRTC 后台服务会依次同步状态 5 和状态 0。
         * @param code 推流结果，0 表示成功，其余值表示出错，详见[错误码](https://cloud.tencent.com/document/product/647/32257)。
         * @param message 具体推流信息。
         * @param extraInfo 扩展信息字段，个别错误码可能会带额外的信息帮助定位问题。
         */
        void onCdnStreamStateChanged(string cdnURL, int status, int code, string message, string extraInfo);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    屏幕分享相关事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /**
         * 9.1 屏幕分享开启的事件回调。
         *
         * 当您通过 {@link startScreenCapture} 等相关接口启动屏幕分享时，SDK 便会抛出此事件回调。
         */
        void onScreenCaptureStarted();

        /**
         * 9.2 屏幕分享暂停的事件回调。
         *
         * 当您通过 {@link pauseScreenCapture} 暂停屏幕分享时，SDK 便会抛出此事件回调。
         * @param reason 原因。
         * - 0：用户主动暂停。
         * - 1：注意此字段的含义在 MAC 和 Windows 平台有稍微差异。屏幕窗口不可见暂停（Mac）。表示设置屏幕分享参数导致的暂停（Windows）。
         * - 2：表示屏幕分享窗口被最小化导致的暂停（仅 Windows）。
         * - 3：表示屏幕分享窗口被隐藏导致的暂停（仅 Windows）。
         * - 4：表示屏幕共享因系统操作而暂停（仅 iOS）。
         */
        void onScreenCapturePaused(int reason);

        /**
         * 9.3 屏幕分享恢复的事件回调。
         *
         * 当您通过 {@link resumeScreenCapture} 恢复屏幕分享时，SDK 便会抛出此事件回调。
         * @param reason 恢复原因。
         * - 0：用户主动恢复。
         * - 1：注意此字段的含义在 MAC 和 Windows 平台有稍微差异。屏幕窗口恢复可见从而恢复分享（Mac）。屏幕分享参数设置完毕后自动恢复（Windows）。
         * - 2：表示屏幕分享窗口从最小化被恢复（仅 Windows）。
         * - 3：表示屏幕分享窗口从隐藏被恢复（仅 Windows）。
         * - 4：表示屏幕共享因系统操作而恢复（仅 iOS）。
         */
        void onScreenCaptureResumed(int reason);

        /**
         * 9.4 屏幕分享停止的事件回调。
         *
         * 当您通过 {@link stopScreenCapture} 停止屏幕分享时，SDK 便会抛出此事件回调。
         * @param reason 停止原因，0：用户主动停止；1：屏幕窗口关闭导致停止；2：表示屏幕分享的显示屏状态变更（如接口被拔出、投影模式变更等）。
         */
        void onScreenCaptureStoped(int reason);

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    本地录制和本地截图的事件回调
        //
        /////////////////////////////////////////////////////////////////////////////////

        /////////////////////////////////////////////////////////////////////////////////
        //
        //                    废弃的事件回调（建议使用对应的新回调）
        //
        /////////////////////////////////////////////////////////////////////////////////
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                    视频数据自定义回调
    //
    /////////////////////////////////////////////////////////////////////////////////

    public interface ITRTCVideoRenderCallback {
        /**
         * 自定义视频渲染回调。
         *
         * 当您设置了本地或者远端的视频自定义渲染回调之后，SDK 就会将原本要交给渲染控件进行渲染的视频帧通过此回调接口抛送给您，以便于您进行自定义渲染。
         * @param frame  待渲染的视频帧信息。
         * @param userId 视频源的 userId，如果是本地视频回调（{@link setLocalVideoRenderDelegate}），该参数可以忽略。
         * @param streamType 频流类型：主路（Main）一般用于承载摄像头画面，辅路（Sub）一般用于承载屏幕分享画面。
         */
        void onRenderVideoFrame(string userId, TRTCVideoStreamType streamType, TRTCVideoFrame frame);
    }

    public interface ITRTCVideoFrameCallback {
        /**
         * SDK 内部 OpenGL 环境已经创建的通知。
         */
        void onGLContextCreated();

        /**
         * 用于对接第三方美颜组件的视频处理回调。
         *
         * 如果您选购了第三方美颜组件，就需要在 TRTCCloud 中设置第三方美颜回调，之后 TRTC 就会将原本要进行预处理的视频帧通过此回调接口抛送给您。
         * 之后您就可以将 TRTC 抛出的视频帧交给第三方美颜组件进行图像处理，由于抛出的数据是可读且可写的，因此第三方美颜的处理结果也可以同步给 TRTC 进行后续的编码和发送。
         * 情况一：美颜组件自身会产生新的纹理。
         * 如果您使用的美颜组件会在处理图像的过程中产生一帧全新的纹理（用于承载处理后的图像），那请您在回调函数中将 dstFrame.textureId 设置为新纹理的 ID：
         * <pre>
         * int onProcessVideoFrame(TRTCVideoFrame srcFrame, TRTCVideoFrame dstFrame);
         * </pre>
         *
         * 情况二：美颜组件需要您提供目标纹理。
         * 如果您使用的第三方美颜模块并不生成新的纹理，而是需要您设置给该模块一个输入纹理和一个输出纹理，则可以考虑如下方案：
         * <pre>
         * int onProcessVideoFrame(TRTCVideoFrame srcFrame, TRTCVideoFrame dstFrame);
         * </pre>
         *
         * @param srcFrame 用于承载 TRTC 采集到的摄像头画面。
         * @param dstFrame 用于接收第三方美颜处理过的视频画面。
         * @note 目前仅支持 OpenGL 纹理方案（ PC 仅支持 TRTCVideoBufferType_Buffer 格式）。
         */
        int onProcessVideoFrame(TRTCVideoFrame srcFrame, TRTCVideoFrame dstFrame);

        /**
         * SDK 内部 OpenGL 环境被销毁的通知。
         */
        void onGLContextDestroy();
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                    音频数据自定义回调
    //
    /////////////////////////////////////////////////////////////////////////////////

    public interface ITRTCAudioFrameCallback {
        /**
         * 本地采集并经过音频模块前处理后的音频数据回调。
         *
         * 当您设置完音频数据自定义回调之后，SDK 内部会把刚采集到并经过前处理(ANS、AEC、AGC）之后的数据，以 PCM 格式的形式通过本接口回调给您。
         * - 此接口回调出的音频时间帧长固定为0.02s，格式为 PCM 格式。
         * - 由时间帧长转化为字节帧长的公式为 `采样率 × 时间帧长 × 声道数 × 采样点位宽`。
         * - 以 TRTC 默认的音频录制格式48000采样率、单声道、16采样点位宽为例，字节帧长为 `48000 × 0.02s × 1 × 16bit = 15360bit = 1920字节`。
         * @param frame PCM 格式的音频数据帧。
         * @note
         * 1. 请不要在此回调函数中做任何耗时操作，由于 SDK 每隔 20ms 就要处理一帧音频数据，如果您的处理时间超过 20ms，就会导致声音异常。
         * 2. 此接口回调出的音频数据是可读写的，也就是说您可以在回调函数中同步修改音频数据，但请保证处理耗时。
         * 3. 此接口回调出的音频数据已经经过了前处理(ANS、AEC、AGC），但**不包含**背景音、音效、混响等前处理效果，延迟较低。
         */
        void onCapturedRawAudioFrame(TRTCAudioFrame frame);

        /**
         * 本地采集并经过音频模块前处理、音效处理和混 BGM 后的音频数据回调。
         *
         * 当您设置完音频数据自定义回调之后，SDK 内部会把刚采集到并经过前处理、音效处理和混 BGM 之后的数据，在最终进行网络编码之前，以 PCM 格式的形式通过本接口回调给您。
         * - 此接口回调出的音频时间帧长固定为0.02s，格式为 PCM 格式。
         * - 由时间帧长转化为字节帧长的公式为`采样率 × 时间帧长 × 声道数 × 采样点位宽`。
         * - 以 TRTC 默认的音频录制格式48000采样率、单声道、16采样点位宽为例，字节帧长为`48000 × 0.02s × 1 × 16bit = 15360bit = 1920字节`。
         * 特殊说明：
         * 您可以通过设置接口中的 `TRTCAudioFrame.extraData` 字段，达到传输信令的目的。由于音频帧头部的数据块不能太大，建议您写入 `extraData` 时，尽量将信令控制在几个字节的大小，如果超过 100 个字节，写入的数据不会被发送。
         * 房间内其他用户可以通过 {@link ITRTCAudioFrameCallback} 中的 `onRemoteUserAudioFrame` 中的 `TRTCAudioFrame.extraData` 字段回调接收数据。
         * @param frame PCM 格式的音频数据帧。
         * @note
         * 1. 请不要在此回调函数中做任何耗时操作，由于 SDK 每隔 20ms 就要处理一帧音频数据，如果您的处理时间超过 20ms，就会导致声音异常。
         * 2. 此接口回调出的音频数据是可读写的，也就是说您可以在回调函数中同步修改音频数据，但请保证处理耗时。
         * 3. 此接口回调出的数据已经经过了前处理(ANS、AEC、AGC）、音效和混 BGM 处理，声音的延迟相比于 {@link onCapturedAudioFrame} 要高一些。
         */
        void onLocalProcessedAudioFrame(TRTCAudioFrame frame);

        /**
         * 混音前的每一路远程用户的音频数据。
         *
         * 当您设置完音频数据自定义回调之后，SDK 内部会把远端的每一路原始数据，在最终混音之前，以 PCM 格式的形式通过本接口回调给您。
         * - 此接口回调出的音频时间帧长固定为0.02s，格式为 PCM 格式。
         * - 由时间帧长转化为字节帧长的公式为`采样率 × 时间帧长 × 声道数 × 采样点位宽`。
         * - 以 TRTC 默认的音频录制格式48000采样率、单声道、16采样点位宽为例，字节帧长为`48000 × 0.02s × 1 × 16bit = 15360bit = 1920字节`。
         * @param frame PCM 格式的音频数据帧。
         * @param userId 用户标识。
         * @note 此接口回调出的音频数据是只读的，不支持修改。
         */
        void onPlayAudioFrame(TRTCAudioFrame frame, string userId);

        /**
         * 将各路待播放音频混合之后并在最终提交系统播放之前的数据回调。
         *
         * 当您设置完音频数据自定义回调之后，SDK 内部会把各路待播放的音频混合之后的音频数据，在提交系统播放之前，以 PCM 格式的形式通过本接口回调给您。
         * - 此接口回调出的音频时间帧长固定为0.02s，格式为 PCM 格式。
         * - 由时间帧长转化为字节帧长的公式为 `采样率 × 时间帧长 × 声道数 × 采样点位宽`。
         * - 以 TRTC 默认的音频录制格式48000采样率、单声道、16采样点位宽为例，字节帧长为 `48000 × 0.02s × 1 × 16bit = 15360bit = 1920字节`。
         * @param frame PCM 格式的音频数据帧。
         * @note
         * 1. 请不要在此回调函数中做任何耗时操作，由于 SDK 每隔 20ms 就要处理一帧音频数据，如果您的处理时间超过 20ms，就会导致声音异常。
         * 2. 此接口回调出的音频数据是可读写的，也就是说您可以在回调函数中同步修改音频数据，但请保证处理耗时。
         * 3. 此接口回调出的是对各路待播放音频数据的混合，但其中并不包含耳返的音频数据。
         */
        void onMixedPlayAudioFrame(TRTCAudioFrame frame);
    }

    /////////////////////////////////////////////////////////////////////////////////
    //
    //                    更多事件回调接口
    //
    /////////////////////////////////////////////////////////////////////////////////

    public interface ITRTCLogCallback {
        /**
         * 本地 LOG 的打印回调。
         *
         * 如果您希望捕获 SDK 的本地日志打印行为，可以通过设置日志回调，让 SDK 将要打印的日志都通过本回调接口抛送给您。
         * @param log 日志内容。
         * @param level 日志等级请参见 {@link TRTC_LOG_LEVEL}。
         * @param module 保留字段，暂无具体意义，目前为固定值 `TXLiteAVSDK`。
         */
        void onLog(string log, TRTCLogLevel level, string module);
    }

}
